#include <QDebug>
#include "treemodel.h"

using namespace Wt;

TreeModel::TreeModel(WObject *parent): WAbstractItemModel(parent)
{
	rootNode = new DomainObjectNode;
}

int
TreeModel::rowCount(const WModelIndex &parent) const
{
	if (parent.column() > 0)
	{
		return 0;
	}

	DomainObjectNode* parentNode = nodeFromIndex(parent);
	if (parentNode == 0)
	{
		return 0;
	}

	return parentNode->childNodes.size();
}

int
TreeModel::columnCount(const WModelIndex &parent) const
{
	Q_UNUSED(parent)
	return 1;
}

boost::any
TreeModel::data(const WModelIndex &index, int role) const
{
	if (role != Qt::DisplayRole && role != Qt::ToolTipRole)
	{
		return boost::any();
	}

	if (index.column() != 0)
	{
		return boost::any();
	}

	DomainObjectNode* node = nodeFromIndex(index);
	if (node == 0)
	{
		return boost::any();
	}

	DomainObject* object = node->object;
	if (object == 0)
	{
		return boost::any();
	}

	return WString::fromUTF8(object->getObjectName().toStdString());
}

boost::any
TreeModel::headerData(int section, Wt::Orientation orientation, int role) const
{
	if (orientation == Wt::Horizontal && section == 0 && role == Wt::DisplayRole)
	{
		return WString("Name");
	}

	return boost::any();
}

DomainObject*
TreeModel::getDomainObject(const WModelIndex &index) const
{
	DomainObjectNode* node = nodeFromIndex(index);
	if (!node)
	{
		return 0;
	}

	return node->object;
}

DomainObjectNode* TreeModel::nodeFromIndex(const WModelIndex& index) const
{
	if (index.isValid())
	{
		return static_cast<DomainObjectNode*>(index.internalPointer());
	}
	else
	{
		return rootNode;
	}
}

/** @reimp */
WModelIndex
TreeModel::index(int row, int column, const WModelIndex& parent) const
{
	if (rootNode == 0 || row < 0 || column < 0)
	{
		return WModelIndex();
	}

	DomainObjectNode* parentNode = nodeFromIndex(parent);
	DomainObjectNode* childNode = parentNode->childNodes.value(row);
	if (childNode == 0)
	{
		return WModelIndex();
	}

	return createIndex(row, column, childNode);
}

/** @reimp */
WModelIndex
TreeModel::parent(const WModelIndex &index) const
{
	QString log("DomainObjectModel::parent()");
	DomainObjectNode* node = nodeFromIndex(index);
	if (node == 0)
	{
		return WModelIndex();
	}

	DomainObjectNode* parentNode = node->parentNode;
	if (parentNode == 0)
	{
		return WModelIndex();
	}

	DomainObjectNode* grandParentNode = parentNode->parentNode;
	if (grandParentNode == 0)
	{
		return WModelIndex();
	}

	int row = grandParentNode->childNodes.indexOf(parentNode);
	if (row < 0)
	{
		qWarning() << log << "Row < 0";
		return WModelIndex();
	}

	return createIndex(row, 0, parentNode);
}

/** @brief Добавить объекты в модель
	@param objects - объекты
*/
void
TreeModel::addObjects(QList<DomainObject*> objects)
{
	QString log("DomainObjectModel::addObjects()");

	Q_FOREACH (DomainObject* object, objects)
	{
		QString id = object->getObjectId();
		if (!id.isEmpty())
		{
			DomainObjectNode* node = new DomainObjectNode;
			node->object = object;

			nodeHash[id] = node;
		}
	}

	Q_FOREACH (DomainObject* object, objects)
	{
		DomainObjectNode* node = nodeHash.value(object->getObjectId(), 0);
		if (node != 0)
		{
			QString parentId = object->getParentObjectId();
			DomainObjectNode* parentNode;
			if (!parentId.isEmpty())
			{
				parentNode = nodeHash.value(parentId, 0);
			}
			else
			{
				parentNode = rootNode;
			}

			if (parentNode != 0)
			{
				WModelIndex parentIndex = getDomainObjectIndex(parentNode->object);
				int row = parentNode->childNodes.size();

				beginInsertRows(parentIndex, row, row);

				node->parentNode = parentNode;
				parentNode->childNodes.append(node);

				endInsertRows();
			}
			else
			{
				qWarning() << log << "DomainObjectNode* parent node is null!";
			}
		}
		else
		{
			qWarning() << log << "DomainObjectNode* node is null!";
		}
	}
}

WModelIndex
TreeModel::getDomainObjectIndex(DomainObject* object) const
{
	WModelIndex rootIndex;
	return searchDomainObject(object, rootIndex);
}

/** @brief Удаляет объекты из модели
	@param objects - объекты
*/
void
TreeModel::removeObjects(QList<DomainObject*> objects)
{
	Q_FOREACH (DomainObject* object, objects)
	{
		removeObject(object);
	}
}

/** @brief Удаляет объект из модели
	@param object - объект
*/
void
TreeModel::removeObject(DomainObject *object)
{
	WModelIndex index = getDomainObjectIndex(object);
	if (index.isValid())
	{
		int row = index.row();
		WModelIndex parentIndex = parent(index);
		beginRemoveRows(parentIndex, row, row);
		DomainObjectNode* node = nodeFromIndex(index);
		DomainObjectNode* parentNode = node->parentNode;
		parentNode->childNodes.removeAt(row);
		//delete node;

		endRemoveRows();
	}
}

WModelIndex
TreeModel::searchDomainObject(DomainObject *object,
									  WModelIndex searchRootIndex) const
{
	int numChildren = rowCount(searchRootIndex);
	for (int i = 0; i < numChildren; i++)
	{
		WModelIndex childIndex = index(i, 0, searchRootIndex);
		DomainObjectNode* node = nodeFromIndex(childIndex);
		if (node->object == object)
		{
			return childIndex;
		}
		else
		{
			WModelIndex res = searchDomainObject(object, childIndex);
			if (res.isValid())
			{
				return res;
			}
		}
	}
	return WModelIndex();
}
