#include "WIdentityProxyModel"

#include <assert.h>

#include <Wt/WReadOnlyProxyModel>
#include <Wt/WStandardItemModel>

namespace Wt {

class WIdentityProxyModel::DefaultModel : public WReadOnlyProxyModel {
public:
  DefaultModel() :
    empty_(this)
  {
    setSourceModel(&empty_);
  }
  static DefaultModel *cur() {
    if (!cur_)
      cur_ = new DefaultModel();
    return cur_;
  }
private:
  WStandardItemModel empty_;
  static DefaultModel *cur_;
};

WIdentityProxyModel::DefaultModel
  *WIdentityProxyModel::DefaultModel::cur_ = 0;

#define S() (sourceModel() ? sourceModel() : DefaultModel::cur())

WIdentityProxyModel::WIdentityProxyModel(WObject *parent) :
  WAbstractProxyModel(parent)
{
}

WIdentityProxyModel::~WIdentityProxyModel()
{
}

int WIdentityProxyModel::columnCount(const WModelIndex &parent) const
{
  return S()->columnCount(mapToSource(parent));
}

WModelIndex WIdentityProxyModel::index(int row, int column, const WModelIndex &parent) const
{
  if (!hasIndex(row, column, parent))
    return WModelIndex();
  const WModelIndex sourceParent = mapToSource(parent);
  const WModelIndex sourceIndex = S()->index(row, column, sourceParent);
  assert(sourceIndex.isValid());
  return mapFromSource(sourceIndex);
}

WModelIndex WIdentityProxyModel::mapFromSource(const WModelIndex &sourceIndex) const
{
  if (!sourceModel() || !sourceIndex.isValid())
    return WModelIndex();

  assert(sourceIndex.model() == sourceModel());
  return createIndex(sourceIndex.row(), sourceIndex.column(), sourceIndex.internalPointer());
}

WModelIndex WIdentityProxyModel::mapToSource(const WModelIndex &proxyIndex) const
{
  if (!sourceModel() || !proxyIndex.isValid())
    return WModelIndex();
  assert(proxyIndex.model() == this);
  return createSourceIndex(proxyIndex.row(), proxyIndex.column(), proxyIndex.internalPointer());
}

WModelIndex WIdentityProxyModel::parent(const WModelIndex &child) const
{
  assert(!child.isValid() || child.model() == this);
  const WModelIndex sourceIndex = mapToSource(child);
  const WModelIndex sourceParent = sourceIndex.parent();
  return mapFromSource(sourceParent);
}

int WIdentityProxyModel::rowCount(const WModelIndex &parent) const
{
  assert(!parent.isValid() || parent.model() == this);
  return S()->rowCount(mapToSource(parent));
}

void WIdentityProxyModel::setSourceModel(WAbstractItemModel *newSourceModel)
{
  if (sourceModel()) {
    for (unsigned int i = 0; i < modelConnections_.size(); ++i)
      modelConnections_[i].disconnect();
    modelConnections_.clear();
  }

  WAbstractProxyModel::setSourceModel(newSourceModel);

  if (newSourceModel) {

    modelConnections_.push_back(newSourceModel->rowsAboutToBeInserted().connect
      (this, &WIdentityProxyModel::sourceRowsAboutToBeInserted));
    modelConnections_.push_back(newSourceModel->rowsInserted().connect
      (this, &WIdentityProxyModel::sourceRowsInserted));
    modelConnections_.push_back(newSourceModel->rowsAboutToBeRemoved().connect
      (this, &WIdentityProxyModel::sourceRowsAboutToBeRemoved));
    modelConnections_.push_back(newSourceModel->rowsRemoved().connect
      (this, &WIdentityProxyModel::sourceRowsRemoved));
    modelConnections_.push_back(newSourceModel->columnsAboutToBeInserted().connect
      (this, &WIdentityProxyModel::sourceColumnsAboutToBeInserted));
    modelConnections_.push_back(newSourceModel->columnsInserted().connect
      (this, &WIdentityProxyModel::sourceColumnsInserted));
    modelConnections_.push_back(newSourceModel->columnsAboutToBeRemoved().connect
      (this, &WIdentityProxyModel::sourceColumnsAboutToBeRemoved));
    modelConnections_.push_back(newSourceModel->columnsRemoved().connect
      (this, &WIdentityProxyModel::sourceColumnsRemoved));
    modelConnections_.push_back(newSourceModel->modelReset().connect
      (this, &WIdentityProxyModel::sourceModelReset));
    modelConnections_.push_back(newSourceModel->dataChanged().connect
      (this, &WIdentityProxyModel::sourceDataChanged));
    modelConnections_.push_back(newSourceModel->headerDataChanged().connect
      (this, &WIdentityProxyModel::sourceHeaderDataChanged));
    modelConnections_.push_back(newSourceModel->layoutAboutToBeChanged().connect
      (this, &WIdentityProxyModel::sourceLayoutAboutToBeChanged));
    modelConnections_.push_back(newSourceModel->layoutChanged().connect
      (this, &WIdentityProxyModel::sourceLayoutChanged));
  }
}

bool WIdentityProxyModel::insertColumns(int column, int count, const WModelIndex &parent)
{
  assert(!parent.isValid() || parent.model() == this);
  return S()->insertColumns(column, count, mapToSource(parent));
}

bool WIdentityProxyModel::insertRows(int row, int count, const WModelIndex &parent)
{
  assert(!parent.isValid() || parent.model() == this);
  return S()->insertRows(row, count, mapToSource(parent));
}

bool WIdentityProxyModel::removeColumns(int column, int count, const WModelIndex &parent)
{
  assert(!parent.isValid() || parent.model() == this);
  return S()->removeColumns(column, count, mapToSource(parent));
}

bool WIdentityProxyModel::removeRows(int row, int count, const WModelIndex &parent)
{
  assert(!parent.isValid() || parent.model() == this);
  return S()->removeRows(row, count, mapToSource(parent));
}

void WIdentityProxyModel::sort(int column, SortOrder order)
{
  S()->sort(column, order);
}

boost::any WIdentityProxyModel::headerData(int section,
					    Orientation orientation,
					    int role) const
{
    return S()->headerData(section, orientation, role);
}

void WIdentityProxyModel::sourceColumnsAboutToBeInserted(const WModelIndex &parent, int start, int end)
{
  assert(!parent.isValid() || parent.model() == sourceModel());
  beginInsertColumns(mapFromSource(parent), start, end);
}

void WIdentityProxyModel::sourceColumnsAboutToBeRemoved(const WModelIndex &parent, int start, int end)
{
  assert(!parent.isValid() || parent.model() == sourceModel());
  beginRemoveColumns(mapFromSource(parent), start, end);
}

void WIdentityProxyModel::sourceColumnsInserted(const WModelIndex &parent, int start, int end)
{
  assert(!parent.isValid() || parent.model() == sourceModel());
  endInsertColumns();
}

void WIdentityProxyModel::sourceColumnsRemoved(const WModelIndex &parent, int start, int end)
{
  assert(!parent.isValid() || parent.model() == sourceModel());
  endRemoveColumns();
}

void WIdentityProxyModel::sourceRowsAboutToBeInserted(const WModelIndex &parent, int start, int end)
{
  assert(!parent.isValid() || parent.model() == sourceModel());
  beginInsertRows(mapFromSource(parent), start, end);
}

void WIdentityProxyModel::sourceRowsAboutToBeRemoved(const WModelIndex &parent, int start, int end)
{
  assert(!parent.isValid() || parent.model() == sourceModel());
  beginRemoveRows(mapFromSource(parent), start, end);
}

void WIdentityProxyModel::sourceRowsInserted(const WModelIndex &parent, int start, int end)
{
  assert(!parent.isValid() || parent.model() == sourceModel());
  endInsertRows();
}

void WIdentityProxyModel::sourceRowsRemoved(const WModelIndex &parent, int start, int end)
{
  assert(!parent.isValid() || parent.model() == sourceModel());
  endRemoveRows();
}

void WIdentityProxyModel::sourceDataChanged(const WModelIndex &topLeft, const WModelIndex &bottomRight)
{
  assert(!topLeft.isValid() || topLeft.model() == sourceModel());
  assert(!bottomRight.isValid() || bottomRight.model() == sourceModel());
  dataChanged().emit(mapFromSource(topLeft), mapFromSource(bottomRight));
}

void WIdentityProxyModel::sourceHeaderDataChanged(Orientation orientation, int start, int end)
{
  headerDataChanged().emit(orientation, start, end);
}

void WIdentityProxyModel::sourceLayoutAboutToBeChanged()
{
  layoutAboutToBeChanged().emit();
}

void WIdentityProxyModel::sourceLayoutChanged()
{
  layoutChanged().emit();
}

void WIdentityProxyModel::sourceModelReset()
{
  modelReset().emit();
}

} // namespace Wt
