#include "DemoModel.h"
#include <Wt/WServer>
#include <Wt/WTime>
#include <boost/range.hpp>
#include <stdexcept>
#include <cassert>

DemoModel::DemoModel(const std::string& sessionId) :
Wt::WAbstractTableModel(),
c_ctorThreadId(std::this_thread::get_id()),
c_sessionId(sessionId),
c_terminateRowInserterThread(false)
{
	// let's add a first row, just for fun...
	addRow(L"First element from Constructor");

	// ...before we start our row inserter thread
	c_rowInserter = std::thread(&DemoModel::rowInserterThreadProc, this);
}

DemoModel::~DemoModel()
{
	// signal our row inserter thread to end...
	c_terminateRowInserterThread = true;

	// ...and wait until it dies
	c_rowInserter.join();
}

int DemoModel::columnCount(const Wt::WModelIndex& parent) const
{
	assert(c_ctorThreadId == std::this_thread::get_id());

	int result = 0;
	if (!(parent.isValid()))
	{
	result = 3; // in this demo we have 3 columns. Always!
	}
	return result;
}

int DemoModel::rowCount(const Wt::WModelIndex& parent) const
{
	assert(c_ctorThreadId == std::this_thread::get_id());

	int result = 0;
	if (!(parent.isValid()))
	{
	std::lock_guard<std::mutex> lock(c_rowsGuard);
	result = c_rows.size(); // return the current count of our rows
	}
	return result;
}

boost::any DemoModel::data(const Wt::WModelIndex& index, int role) const
{
	assert(c_ctorThreadId == std::this_thread::get_id());

	boost::any result;

	if(role == Wt::DisplayRole)
	{
		std::lock_guard<std::mutex> lock(c_rowsGuard);

		const int row = index.row();
		if (row >= 0 && row < int(c_rows.size()))
		{
			switch (index.column())
			{
				case 0:
				{
					// our ascending number column
					result = c_rows.at(row).number;
					break;
				}
				case 1:
				{
					// our custom message column
					result = c_rows.at(row).message;
					break;
				}
				case 2:
				{
					// our insert time column
					result = c_rows.at(row).insertTime;
					break;
				}
			}
		}
	}

	return result;
}

boost::any DemoModel::headerData(int section, Wt::Orientation orientation, int role) const
{
	assert(c_ctorThreadId == std::this_thread::get_id());

	boost::any result;

	static const wchar_t* const columnNames[] =
	{
		L"Number", L"Message", L"Insert Time"
	};
	if (orientation == Wt::Horizontal && role == Wt::DisplayRole && section >= 0 && section < int(boost::size(columnNames)))
	{
		result = Wt::WString(columnNames[section]);
	}

	return result;
}

void DemoModel::addRow(const std::wstring& msg)
{
	std::lock_guard<std::mutex> lock(c_rowsGuard);
	c_rows.push_back(SRow(msg));
}

void DemoModel::rowInserterThreadProc()
{
	// we need a server instance, so we can inform our application about row updates
	const auto server = Wt::WServer::instance();
	assert(server);
	if (server)
	{
		// our main loop
		while (!c_terminateRowInserterThread)
		{
			// let's add a new row...
			addRow(L"Row from inserter thread.");

			// ...and inform our application about this
			server->post(c_sessionId, boost::bind(&DemoModel::postLayoutChanged, this));

			// let's sleep for a second until we insert the next row
			std::this_thread::sleep_for(std::chrono::seconds(1));
		}
	}
}

void DemoModel::postLayoutChanged()
{
	// We are called within the context of Wt::WServer::post() which was placed in DemoModel::rowInserterThreadProc()
	const auto app  = Wt::WApplication::instance();
	assert(app);
	if (app)
	{
		// So here is issue #1: to my understanding emitting layoutChanged is sufficient to
		// update the table view immediately...
		layoutChanged().emit();

		// ...but it does not happen until the user is doing some action inside the browser, e.g.
		// by clicking on a row.

		// Anyway, if I call tiggerUpdate() the user is getting the model updates in his browser as expected, but
		// this brings us to issue #2:
		// The resulting update is then not done from within the WQapplication thread context anymore.
		// Watch out for the asserts in columnCount(), rowCount(), data() and headerDate().

		// comment out the next line to see issue #2 in action
		app->triggerUpdate();
	}
}


// Tricky implementation of our row data :-)
int DemoModel::SRow::sequence = 0;

DemoModel::SRow::SRow(const std::wstring& msg) :
number(Wt::WString(L"{1}").arg(sequence++)),
message(msg),
insertTime(Wt::WTime::currentServerTime().toString(L"HH:mm:ss (zzz)"))
{
}

