#ifndef DEMOMODEL__H
#define DEMOMODEL__H

#include <Wt/WAbstractTableModel>
#include <string>
#include <vector>
#include <ctime>
#include <mutex>
#include <thread>
#include <atomic>

namespace WB
{
		// This class implements a model whose data is updated by its private thread
		// In this demo there is simply a new row inserted every second
		// The Constructor of this class remember the thread id it is called from
		// and it will assert if one of the Wt::WAbstractTableModel methods are called
		// from another thread. As our DemoApplication creates us within
		// Wt::WQApplication::create() this should be always the case.
		class DemoModel : public Wt::WAbstractTableModel
		{
				DemoModel(const DemoModel&) = delete;
				DemoModel& operator = (const DemoModel&) = delete;

				public:
						// We need a session id so our row inserter thread knows who to inform about its model updates
						DemoModel(const std::string& sessionId);

						// our destructor simply stops the row inserter thread
						virtual ~DemoModel();

						// Wt::WAbstractTableModel stuff
						virtual int columnCount(const Wt::WModelIndex& parent) const override;
						virtual int rowCount(const Wt::WModelIndex& parent) const override;
						virtual boost::any data(const Wt::WModelIndex& index, int role) const override;
						virtual boost::any headerData(int section, Wt::Orientation orientation, int role) const override;

						// This adds a row to our model data.
						// The first column is a ascending decimal value, the second is the message passed as parameter 
						// and the third is the current time.
						// It is safe to call this from multiple threads.
						void addRow(const std::wstring& msg);
	
				private:
						// Internal helper struct for our Rows
						struct SRow
						{
								SRow(const std::wstring& msg);
								static int sequence;     // to generate ascending values from

								int number;              // a value from our static sequence
								std::wstring message;    // a custom message
								std::time_t insertTime;  // time when this row was created
						};

						// This is the entry point of our private rowInserter-Thread
						void rowInserterThreadProc();

						// This is simply a helper function to call layoutChanged().emit() and it 
						// will assert if it is not called from a valid Wt::WApplication-thread-context
						void postLayoutChanged();


						// We remember the thread id of our constructor as we are created 
						// from Wt::WQApplication::create()
						const std::thread::id c_ctorThreadId;

						// We rely in this demo, that our session id will never change
						const std::string c_sessionId;

						// Lock this mutex, if you access c_rows directly
						mutable std::mutex c_rowsGuard;

						// our row data. You need to lock c_rowsGuard before accessing it
						std::vector<SRow> c_rows;

						// our row inserter thread
						std::thread c_rowInserter;

						// our row inserter thread will run as long as this value is false
						std::atomic<bool> c_terminateRowInserterThread;
		};
}

#endif
