#include <boost/algorithm/string.hpp>
#include <Wt/WApplication>
#include <Wt/WStandardItemModel>
#include <Wt/WStandardItem>
#include <Wt/WPushButton>
#include <Wt/WTableView>
#include <Wt/Dbo/backend/Sqlite3>
#include <Wt/Dbo/QueryModel>
#include <Wt/Dbo/FixedSqlConnectionPool>

using namespace Wt;

class DboTestRecord {
  public:
    DboTestRecord() = default;
    DboTestRecord(std::string row_string) : rowString(row_string) { }

    std::string rowString;

    template<class Action>
    void persist(Action& a)
    {
      Dbo::field(a, rowString, "row_string");
    }
};

class ConnectPool {
  public:
    ConnectPool()
    {
      connect_pool_.reset(new Dbo::FixedSqlConnectionPool(
          new Dbo::backend::Sqlite3(":memory:") , 5));
    }

    static Dbo::FixedSqlConnectionPool *connectPool() { return connect_pool_.get(); }

  private:
    static std::unique_ptr<Dbo::FixedSqlConnectionPool> connect_pool_;
};
std::unique_ptr<Dbo::FixedSqlConnectionPool> ConnectPool::connect_pool_;

class Session : public Dbo::Session
{
public:
  Session()
  {
    setConnectionPool(*ConnectPool::connectPool());
    mapClass<DboTestRecord>("test_table");
  }
};

class TestApp : public WApplication
{
public:
  TestApp(const WEnvironment& env) : WApplication(env)
  {
    Dbo::Transaction transaction(session_);

    new WText("NOTE: Pressing the 'Modify Headers' button should update<br/>"
      "the column header in both tables to display '*MODIFIED*'", root());

    auto standardItemModel = new WStandardItemModel(this);
    std::vector<WStandardItem *> rows = {
      new WStandardItem("row 1"), new WStandardItem("row 2")};
    standardItemModel->appendColumn(rows);
    standardItemModel->setHeaderData(0, "Standard Item Model Unmodified");
    models_.push_back(standardItemModel);

    auto queryModel = new Dbo::QueryModel<Dbo::ptr<DboTestRecord> >(this);
    queryModel->setQuery(session_.find<DboTestRecord>());
    queryModel->addColumn("row_string", "Query Model Unmodified");
    models_.push_back(queryModel);

    for (auto m: models_) {
      WTableView *tableView = new WTableView(root());
      tableView->setModel(m);
      tableView->sortByColumn(0, AscendingOrder);
      tableView->setColumnWidth(0, 280);
      tableView->setMargin(20);
    }

    (new WPushButton("Modify Headers", root()))->clicked().connect(std::bind([=] {
      for (auto m: models_) 
        m->setHeaderData(0, Horizontal, boost::replace_first_copy(
            asString(m->headerData(0, Horizontal)).narrow(), "Unmodified", "*MODIFIED*"));
    }));
  }

private:
  Session session_;
  std::vector<WAbstractItemModel *> models_;
};

int main(int argc, char *argv[])
{
  ConnectPool global_pool;
  {
    Session session;
    Dbo::Transaction transaction(session);

    session.createTables();
    session.add(new DboTestRecord("row 1"));
    session.add(new DboTestRecord("row 2"));
  }

  return WRun(argc, argv, [](const WEnvironment& env) {return new TestApp(env);});
}
