#include <boost/algorithm/string.hpp>
#include <Wt/WApplication>
#include <Wt/WStandardItemModel>
#include <Wt/WSortFilterProxyModel>
#include <Wt/WStandardItem>
#include <Wt/WPushButton>
#include <Wt/WTableView>
#include <Wt/Dbo/backend/Sqlite3>
#include <Wt/Dbo/QueryModel>
#include <Wt/Dbo/FixedSqlConnectionPool>

using namespace Wt;

class DboTestRecord {
  public:
    DboTestRecord() = default;
    DboTestRecord(std::string row_string) : rowString(row_string) { }

    std::string rowString;

    template<class Action>
    void persist(Action& a)
    {
      Dbo::field(a, rowString, "row_string");
    }
};

class ConnectPool {
  public:
    ConnectPool()
    {
      connect_pool_.reset(new Dbo::FixedSqlConnectionPool(
          new Dbo::backend::Sqlite3(":memory:") , 5));
    }

    static Dbo::FixedSqlConnectionPool *connectPool() { return connect_pool_.get(); }

  private:
    static std::unique_ptr<Dbo::FixedSqlConnectionPool> connect_pool_;
};
std::unique_ptr<Dbo::FixedSqlConnectionPool> ConnectPool::connect_pool_;

class Session : public Dbo::Session
{
public:
  Session()
  {
    setConnectionPool(*ConnectPool::connectPool());
    mapClass<DboTestRecord>("test_table");
  }
};

class TestApp : public WApplication
{
public:
  TestApp(const WEnvironment& env) : WApplication(env)
  {
    Dbo::Transaction transaction(session_);

    new WText("NOTE: Pressing 'Update Models' button will update the two<br/>"
      "models backing the displayed WTableViews. The sort<br/>"
      "indicator and sort order should remain unchanged.", root());

    auto standardItemModel = new WStandardItemModel(this);
    std::vector<WStandardItem *> rows = {
      new WStandardItem("row 1"), new WStandardItem("row 2"), new WStandardItem("row 3")};
    standardItemModel->appendColumn(rows);

    auto sortFilterProxyModel = new WSortFilterProxyModel(this);
    sortFilterProxyModel->setSourceModel(standardItemModel);
    sortFilterProxyModel->setHeaderData(0, Horizontal, "WSortFilterProxyModel on WStandardItem Model");
    sortFilterProxyModel->setFilterKeyColumn(0);
    models_.push_back(sortFilterProxyModel);

    auto queryModel = new Dbo::QueryModel<Dbo::ptr<DboTestRecord> >(this);
    queryModel->setQuery(session_.find<DboTestRecord>());
    queryModel->addColumn("row_string", "Query Model");
    models_.push_back(queryModel);

    for (auto m: models_) {
      WTableView *tableView = new WTableView(root());
      tableView->setModel(m);
      tableView->sortByColumn(0, DescendingOrder);
      tableView->setColumnWidth(0, 360);
      tableView->setMargin(20);
    }

    (new WPushButton("Update Models", root()))->clicked().connect(std::bind([=] {
      Dbo::Transaction transaction2(session_);
      queryModel->setQuery(session_.find<DboTestRecord>().where("id != 2"), true);
      sortFilterProxyModel->setFilterRegExp("row [13]");
    }));
  }

private:
  Session session_;
  std::vector<WAbstractItemModel *> models_;
};

int main(int argc, char *argv[])
{
  ConnectPool global_pool;
  {
    Session session;
    Dbo::Transaction transaction(session);

    session.createTables();
    session.add(new DboTestRecord("row 1"));
    session.add(new DboTestRecord("row 2"));
    session.add(new DboTestRecord("row 3"));
  }

  return WRun(argc, argv, [](const WEnvironment& env) {return new TestApp(env);});
}
