#include "WSpectrogramChart.h"
#include <Wt/WPainter>
#include <Wt/WContainerWidget>

using namespace Wt;

namespace
{
   //! Size of the Z-Axis
   const unsigned ZSIZE = 20;
   //! Chart title size
   const unsigned TSIZE = 30;
}

namespace Wt {
   namespace Chart {

WSpectrogramChart::WSpectrogramChart( WContainerWidget * parent )
   : Wt::WPaintedWidget(parent),
     plotImage_( new WRasterImage("png",WLength(256),WLength(256), parent ) ),
     zImage_( new WRasterImage("png",WLength(256),WLength(1), parent ) ),
     colorMap_(0),
     data_(0)
{
   scale_[X_AXIS] = 0;
   scale_[Y_AXIS] = 0;
   scale_[Z_AXIS] = 0;
   scaleEngine_[X_AXIS] = 0;
   scaleEngine_[Y_AXIS] = 0;
   scaleEngine_[Z_AXIS] = 0;
}

WSpectrogramChart::~WSpectrogramChart()
{
   delete plotImage_;
}

const WString & WSpectrogramChart::title()
{
   return title_;
}

void WSpectrogramChart::setTitle(const WString & t)
{
   title_ = t;
}

WColorMap & WSpectrogramChart::colorMap()
{
   return *colorMap_;
}

void WSpectrogramChart::setColorMap(WColorMap & map)
{
   colorMap_ = &map;
}

WAbstractSpectrogramData & WSpectrogramChart::data()
{
   return *data_;
}

void WSpectrogramChart::setData(WAbstractSpectrogramData& dat)
{
   data_ = &dat;
}

WAbstractScale * WSpectrogramChart::scale(Axis axis)
{
   return scale_[axis];
}

void WSpectrogramChart::setScale(Axis axis, WAbstractScale * scale)
{
   scale_[axis] = scale;
}

WAbstractScaleEngine * WSpectrogramChart::scaleEngine(Axis axis)
{
   return scaleEngine_[axis];
}

void WSpectrogramChart::setScaleEngine(Axis axis, WAbstractScaleEngine * engine)
{
   scaleEngine_[axis] = engine;
}

void WSpectrogramChart::paintEvent(Wt::WPaintDevice* paintDevice)
{
   WPainter painter(paintDevice);
   painter.setRenderHint(WPainter::Antialiasing);
   paint(painter);
}

void WSpectrogramChart::paint(Wt::WPainter& painter)
{
   WRectF area = painter.window();
   WRectF plotArea = computePlotArea(area);
   WLength plotareaWidth  = plotArea.width();
   WLength plotareaHeight = plotArea.height();
   if( plotImage_->width() != plotareaWidth || plotImage_->height() != plotareaHeight )
   {
      //FIXME: This will crash if deleted while being served
      delete plotImage_;
      plotImage_ = new WRasterImage("png",plotareaWidth,plotareaHeight,(WObject*)this);
      delete zImage_;
      zImage_ = new WRasterImage("png",plotareaWidth,WLength(1),(WObject*)this );
   }

   drawPlot( plotareaWidth.toPixels(), plotareaHeight.toPixels() );
   drawZ( plotareaWidth.toPixels() );

   drawAxes(painter, plotArea);

   WFont font;
   font.setSize( WFont::FixedSize, TSIZE );
   painter.setFont( font );
   painter.drawText(0,0,area.right(),1,AlignCenter|AlignTop,title_);

   painter.drawImage( plotArea.left(), plotArea.top(), WPainter::Image(plotImage_->generateUrl(),plotareaWidth.toPixels(),plotareaHeight.toPixels()) );
   painter.drawImage( WRectF(plotArea.left(), plotArea.bottom() + scale_[X_AXIS]->computedDepth().toPixels(),plotareaWidth.toPixels(),ZSIZE), WPainter::Image(zImage_->generateUrl(),plotareaWidth.toPixels(),1) );
}

void WSpectrogramChart::drawAxes(WPainter& painter, WRectF& plotArea)
{
   WRectF rect = data_->boundingRect();
   scale_[X_AXIS]->draw(painter, plotArea, scaleEngine_[X_AXIS]->generateTicks(rect.left(),rect.right(),plotArea.width()) );
   scale_[Y_AXIS]->draw(painter, plotArea, scaleEngine_[Y_AXIS]->generateTicks(rect.top(),rect.bottom(),plotArea.height()) );
   WRectF zrect;
   zrect.setX( plotArea.left() );
   zrect.setY( plotArea.bottom() + scale_[X_AXIS]->computedDepth().toPixels() );
   zrect.setWidth( plotArea.width() );
   zrect.setHeight( ZSIZE );

   double min, max;
   data_->range(min,max);
   scale_[Z_AXIS]->draw(painter, zrect, scaleEngine_[Z_AXIS]->generateTicks( min, max, zrect.width() ) );
}

void WSpectrogramChart::drawPlot(unsigned cols, unsigned rows) const
{
   WRectF rect = data_->boundingRect();

   double min, max;
   data_->range(min,max);

   for(unsigned x = 0; x < cols; ++x)
   {
      for(unsigned y = 0; y < rows; ++y)
      {
         double xt = scaleEngine_[X_AXIS]->invTransform( x, 0, cols, rect.left(), rect.right() );
         double yt = scaleEngine_[Y_AXIS]->invTransform( y, 0, rows, rect.top(), rect.bottom() );
         double raw = data_->value( xt, yt );
         double value = scaleEngine_[Z_AXIS]->transform( raw, min, max, min, max );
         WColor color = colorMap_->color( value, min, max );
         plotImage_->setPixel(x,rows-y-1,color);
      }
   }
}

void WSpectrogramChart::drawZ(unsigned cols) const
{
   for(unsigned x = 0; x < cols; ++x)
   {
      double value = scaleEngine_[Z_AXIS]->transform( x, 0, cols, 0, cols );
      WColor color = colorMap_->color( value, 0, cols );
      zImage_->setPixel(x,0,color);
   }
}

WRectF WSpectrogramChart::computePlotArea(WRectF area)
{
   scale_[X_AXIS]->setOrientation(WAbstractScale::South);
   scale_[Y_AXIS]->setOrientation(WAbstractScale::West);
   scale_[Z_AXIS]->setOrientation(WAbstractScale::South);

   double left   = area.left() + scale_[Y_AXIS]->computedDepth().toPixels();
   double right  = area.right() - scale_[Y_AXIS]->computedDepth().toPixels();
   double top    = area.top() + TSIZE + 5;
   double bottom = area.bottom() - scale_[X_AXIS]->computedDepth().toPixels() - ZSIZE - scale_[Z_AXIS]->computedDepth().toPixels();

   return WRectF( left, top, right-left, bottom-top );
}

void WSpectrogramChart::layoutSizeChanged(int width, int height)
{
}

void WSpectrogramChart::resize(const WLength & width, const WLength & height)
{
   WPaintedWidget::resize(width,height);
}

void WSpectrogramChart::resizePlotArea(const Wt::WLength & width, const Wt::WLength & height)
{
   WRectF plotarea = computePlotArea(WRectF(0,0,0,0));
   resize( width.toPixels() - plotarea.width(), height.toPixels() - plotarea.height() );
}

   }
}

