#pragma once

#include <Wt/WObject>

namespace Wt {
   namespace Chart {

//! \brief Baseclass for an axis scaling engine
//! E.g. Linear Scaling, Log_x scaling
class WAbstractScaleEngine : public Wt::WObject
{
public:
   WAbstractScaleEngine( Wt::WObject * parent = 0 ) : Wt::WObject(parent) {}
   virtual ~WAbstractScaleEngine() {}

   typedef struct{ unsigned mark; double pos; double value; } TickMark;

   //! \brief A series of tick marks
   typedef std::vector< TickMark > Ticks;

   //! \brief Determine the ideal location for ticks/labels with the current scaling
   //!
   //! \param min        Minium number of tick marks
   //! \param max        Maximum number of tick marks
   //! \param axisLength Size of the axis
   //!
   //! \return Tick marks to draw
   virtual Ticks generateTicks(double min, double max, const Wt::WLength& axisLength) = 0;

   //! \brief Transfroms the input point into its location in the scaled environment
   //!
   //! The input range is typically the range of the input data for an axis. The output
   //! range would be the range to be plotted on, probably in pixels. The return value
   //! would then be the pixel for the input point.
   //!
   //! \param in   The input point to be transformed
   //! \param iMin Minimum value for in
   //! \param iMax Maximum value for in
   //! \param oMin Minimum value for the output point
   //! \param oMax Maximum value for the output point
   //!
   //! \returns output point
   //!
   //! \sa WAbstractScaleEngine::invTransform
   //!
   virtual double transform( const double& in, const double& iMin, const double& iMax, const double& oMin, const double& oMax) = 0;

   //! \brief Performs the inverse tranformation
   //!
   //! The input range is typically the range of the plot, probably in pixels. The output
   //! range would be the range of the plot data. The return value would then be the location
   //! in the data for the input pixel.
   //!
   //! \note If implemented correctly a =~ invTransform( transform(a, i, j, k, l), k, l, i, j )
   //!
   //! \param in   The input point to be transformed
   //! \param iMin Minimum value for in
   //! \param iMax Maximum value for in
   //! \param oMin Minimum value for the output point
   //! \param oMax Maximum value for the output point
   //!
   //! \returns output point
   //!
   //! \sa WAbstractScaleEngine::transform
   //!
   virtual double invTransform( const double& in, const double& iMin, const double& iMax, const double& oMin, const double& oMax) = 0;
};

   }
}

