#include <signal.h>
#include <unistd.h>
#include <Wt/Http/Request.h>
#include <Wt/Http/Response.h>
#include <Wt/WApplication.h>
#include <Wt/WBootstrapTheme.h>
#include <Wt/WEnvironment.h>
#include <Wt/WMemoryResource.h>
#include <Wt/WServer.h>
#include <Wt/WStandardItem.h>
#include <Wt/WStandardItemModel.h>
#include <Wt/WTableView.h>
#include <Wt/WText.h>
#include <Wt/WVBoxLayout.h>


using namespace Wt;

class TableView : public WTableView {
public:
  TableView()
  {
    setLayoutSizeAware(true);
    scrolled().connect([=] (WScrollEvent se) {
      log("info") << "TableView scrolled: scrollX: " << se.scrollX() << ", scrollY: " << se.scrollY()
                  << ", viewportWidth: " << se.viewportWidth() << ", viewportHeight: " << se.viewportHeight();
    });
  }

  void layoutSizeChanged(int width, int height) override
  {
    log("info") << "TableView layoutSizeChanged: (w = " << width << ", h = " << height << ")";
    WTableView::layoutSizeChanged(width, height);
  }
};

class MemoryResource : public Wt::WMemoryResource {
public:
  MemoryResource(const std::string& mimeType, const std::vector<unsigned char>& data) :
      WMemoryResource(mimeType, data) { };

protected:
  void handleRequest(const Http::Request &request, Http::Response &response) override {
    usleep(100000);
    WMemoryResource::handleRequest(request, response);
  };
};

class TestApp : public WApplication {
public:
  TestApp(const WEnvironment& env);

private:
  WText* text_ = nullptr;
};

TestApp::TestApp(const WEnvironment& env) : WApplication(env)
{
  setTitle("Sequence error waitingForJavaScript");

  auto bs_theme = std::make_shared<WBootstrapTheme>();
  bs_theme->setVersion(BootstrapVersion::v3);
  setTheme(bs_theme);

  auto topLayout = std::make_unique<WVBoxLayout>();
  topLayout->setPreferredImplementation(LayoutImplementation::JavaScript);

  auto model = std::make_shared<WStandardItemModel>(2000, 4);
  for (int r = 0; r < model->rowCount(); ++r) {
    for (int c = 0; c < model->columnCount(); ++c) {
      model->setData(model->index(r, c), r + c);
    }
  }

  auto tableView = std::make_unique<TableView>();
  auto tv = tableView.get();
  tableView->setModel(model);
  topLayout->addWidget(std::move(tableView), 1);

  auto text = std::make_unique<WText>("<span style=\"font-size: 30px\">Text Content</span>");
  text_ = text.get();
  topLayout->addWidget(std::move(text), 0);

  root()->setLayout(std::move(topLayout));
  enableUpdates();

  auto server = env.server();
  server->post(sessionId(), [=] () {
    root()->addStyleClass("container-fluid");
    require("js/slow1.js");
    tv->scrollTo(model->index(200, 0));
    text_->setText("<span style=\"font-size: 90px\">Larger Text Content</span>");
    triggerUpdate();
  });
}

int main(int argc, char **argv)
{
  try {
    std::string slow_js_s = "console.log(\"slow.js here\");\n";
    std::vector<unsigned char> slow_js(slow_js_s.begin(), slow_js_s.end());
    MemoryResource jsResource("text/javascript", slow_js);

    WServer server(argv[0]);
    server.setServerConfiguration(argc, argv, WTHTTP_CONFIGURATION);
    server.addResource(&jsResource, "/js");
    server.addEntryPoint(Wt::EntryPointType::Application,
            [] (const WEnvironment& env) { return std::make_unique<TestApp>(env); });
    if (server.start()) {
      int sig = WServer::waitForShutdown();
      std::cerr << "Shutdown (signal = " << sig << ")" << std::endl;
      server.stop();
      if (sig == SIGHUP)
        WServer::restart(argc, argv, nullptr);
    }
  } catch (WServer::Exception& e) {
    std::cerr << e.what() << "\n";
    return 1;
  } catch (std::exception& e) {
    std::cerr << "exception: " << e.what() << "\n";
    return 1;
  }
}
