
#ifndef _DEBUG
#pragma comment(lib, "wt.lib")
#pragma comment(lib, "wthttp.lib")
#else
#pragma comment(lib, "wtd.lib")
#pragma comment(lib, "wthttpd.lib")
#endif

#include <Wt/WServer.h>
#include <Wt/WApplication.h>
#include <Wt/WContainerWidget.h>
#include <Wt/WText.h>
#include <Wt/WVBoxLayout.h>
#include <Wt/WCompositeWidget.h>
#include <Wt/WPushButton.h>
#include <Wt/WDialog.h>

#include <iostream>
#include <memory>
#include <thread>
#include <random>
#include <atomic>
#include <vector>


// CMD:
// C:> server.exe --http-address 0.0.0.0 --docroot C:\Users\user\source\repos\emweb\wt-install\share\Wt


class MyApp : public Wt::WApplication {

private:
   Wt::WText* text{};
   std::atomic<int> count{};
   int dlgCount{};
   std::vector<std::thread> threads;

public:

   static std::unique_ptr<Wt::WApplication> CreateApp (const Wt::WEnvironment& env) 
   {
      return std::make_unique<MyApp>(env);
   }

   void OpenDialog ()
   {
      auto dlg = addChild(std::make_unique<Wt::WDialog>("This dialog should be unique, but it's count is: " + std::to_string(++dlgCount)));

      dlg->resize(450, -1);
      dlg->setModal(true);
      dlg->setClosable(true);
      dlg->finished().connect([&] () { removeChild(dlg); --dlgCount; });

      dlg->show();
   }

   void Work () 
   {
      const auto sessionid = wApp->sessionId();

      const auto finishcb = bindSafe([&] () {
         std::this_thread::sleep_for(std::chrono::seconds(5)); // simulate a slow responding server

         if (--count == 0) {
            text->setText("done");
         }
         else {
            text->setText("loading: " + std::to_string(count));
         }
         
         wApp->triggerUpdate(); 
         wApp->enableUpdates(false);
      });

      constexpr int CONCURRENCY = 1;
      for (int i = 0; i < CONCURRENCY; ++i) {
         ++count;
         wApp->enableUpdates(true);

         threads.push_back(std::thread{[sessionid, finishcb] () {
            Wt::WServer::instance()->post(sessionid, finishcb);
         }});
      }
   }

   ~MyApp () 
   {
      for (auto& t: threads) {
         t.join();
      }
   }

   MyApp (const Wt::WEnvironment& env) 
   : Wt::WApplication(env) 
   {
      auto layout = root()->setLayout(std::make_unique<Wt::WVBoxLayout>());
           
      auto btn = layout->addWidget(std::make_unique<Wt::WPushButton>("open dialog"));
      btn->clicked().connect(this, &MyApp::OpenDialog);

      text = layout->addWidget(std::make_unique<Wt::WText>("warming up, smash dialog button to produce error!"));

      layout->addStretch(1);


      Work();
   }

};


int main(int argc, char** argv)
try {
   Wt::WServer wserver{argc, argv};

   wserver.addEntryPoint(Wt::EntryPointType::Application, &MyApp::CreateApp);

   wserver.run();
   return wserver.waitForShutdown();
}
catch (std::exception& e) {
   std::cerr << e.what() << std::endl;
   return EXIT_FAILURE;
}

