/*
 * Copyright (C) 2008 Emweb bvba, Heverlee, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <Wt/WApplication>
#include <Wt/WBreak>
#include <Wt/WContainerWidget>
#include <Wt/WLineEdit>
#include <Wt/WPushButton>
#include <Wt/WText>
#include <Wt/WResource>
#include <Wt/Http/ResponseContinuation>
#include <Wt/Http/Response>
#include <Wt/WAnchor>
#include <Wt/WServer>

using namespace Wt;

class HelloApplication : public WApplication
{
public:
  HelloApplication(const WEnvironment& env);

private:
  WLineEdit *nameEdit_;
  WText *greeting_;

  void greet();
};

class MyResource : public Wt::WResource
{
public:
  MyResource(int iterations, Wt::WObject *parent = 0)
    : Wt::WResource(parent),
      iterations_(iterations)
  {
    suggestFileName("data.txt");
  }

  ~MyResource() {
    beingDeleted();
  }

  void handleRequest(const Wt::Http::Request& request,
                     Wt::Http::Response& response) {
#if 0
    // see if this request is for a continuation:
    Wt::Http::ResponseContinuation *continuation = request.continuation();

    // calculate the current start
    int iteration = continuation ? boost::any_cast<int>(continuation->data()) : 0;
    if (iteration == 0)
      response.setMimeType("plain/text");

    int last = std::min(iterations_, iteration + 2);
    for (int i = iteration; i < last; ++i)
      response.out() << "Data item " << i << "\r\n";

    // if we have not yet finished, create a continuation to serve more
    if (last < iterations_) {
      continuation = response.createContinuation();
      // remember what to do next
      continuation->setData(last);
    }
#else
                              // see if this request is for a continuation:
       Wt::Http::ResponseContinuation *continuation = request.continuation();

       // calculate the current start
       int iteration = continuation ? boost::any_cast<int>(continuation->data()) : 0;
       if (iteration == 0)
               response.setMimeType("application/xml");

       int last = (std::min)(10, iteration + 2);
       std::cout << "RSSFeed: last: " << last <<  std::endl;

       for (int i = iteration; i < last; ++i)
       {
               response.out() << "Data item " << i << std::endl;
               std::cout << "RSSFeed: " << i <<  std::endl;
       }

       // if we have not yet finished, create a continuation to serve more
       if (last < 10) {
               std::cout << "RSSFeed: continuation" <<  std::endl;
               continuation = response.createContinuation();
               // remember what to do next
               continuation->setData(last);
       }
#endif
  }

private:
  int iterations_;
};

/*
 * The env argument contains information about the new session, and
 * the initial request. It must be passed to the WApplication
 * constructor so it is typically also an argument for your custom
 * application constructor.
*/
HelloApplication::HelloApplication(const WEnvironment& env)
  : WApplication(env)
{
  setTitle("Hello world");                               // application title


  root()->addWidget(new WText("Your name, please ? "));  // show some text
  nameEdit_ = new WLineEdit(root());                     // allow text input
  nameEdit_->setFocus();                                 // give focus

  new WAnchor(new MyResource(10, root()), "click here", root());
  WPushButton *b = new WPushButton("Greet me.", root()); // create a button
  b->setMargin(5, Left);                                 // add 5 pixels margin

  root()->addWidget(new WBreak());                       // insert a line break

  greeting_ = new WText(root());                         // empty text

  /*
   * Connect signals with slots
   *
   * - simple Wt-way
   */
  b->clicked().connect(this, &HelloApplication::greet);

  /*
   * - using an arbitrary function object (binding values with boost::bind())
   */
  nameEdit_->enterPressed().connect
    (boost::bind(&HelloApplication::greet, this));

  /*
   * - using a c++0x lambda:
   */
  // b->clicked().connect(std::bind([=]() { 
  //       greeting_->setText("Hello there, " + nameEdit_->text());
  // }));
}

void HelloApplication::greet()
{
  /*
   * Update the text, using text input into the nameEdit_ field.
   */
  greeting_->setText("Hello there, " + nameEdit_->text());

}

WApplication *createApplication(const WEnvironment& env)
{
  /*
   * You could read information from the environment to decide whether
   * the user has permission to start a new application
   */
  return new HelloApplication(env);
}

namespace {
int MyRun(int argc, char *argv[], ApplicationCreator createApplication)
{
  try {
    // use argv[0] as the application name to match a suitable entry
    // in the Wt configuration file, and use the default configuration
    // file (which defaults to /etc/wt/wt_config.xml unless the environment
    // variable WT_CONFIG_XML is set)
    WServer server(argv[0]);

    // WTHTTP_CONFIGURATION is e.g. "/etc/wt/wthttpd"
    server.setServerConfiguration(argc, argv, WTHTTP_CONFIGURATION);

    server.addResource(new MyResource(100), "/feed/rss2");
    // add a single entry point, at the default location (as determined
    // by the server configuration's deploy-path)
    server.addEntryPoint(Wt::Application, createApplication);
    if (server.start()) {
      while(true) sleep(1000);
    }
  } catch (WServer::Exception& e) {
    std::cerr << e.what() << "\n";
    return 1;
  } catch (std::exception& e) {
    std::cerr << "exception: " << e.what() << "\n";
    return 1;
  }
}
}
int main(int argc, char **argv)
{
  /*
   * Your main method may set up some shared resources, but should then
   * start the server application (FastCGI or httpd) that starts listening
   * for requests, and handles all of the application life cycles.
   *
   * The last argument to WRun specifies the function that will instantiate
   * new application objects. That function is executed when a new user surfs
   * to the Wt application, and after the library has negotiated browser
   * support. The function should return a newly instantiated application
   * object.
   */
  return MyRun(argc, argv, &createApplication);
}

