
#pragma once

#if defined (_MSC_VER)
#	include <windows.h>
#	undef min
#	undef max
#elif defined (__GNUC__)
#	include <chrono>
#	include <time.h>
#endif

namespace sk::tools
{
#if defined (_MSC_VER)

	struct Timer
	{
		//!
		//! A constructor
		//!
		Timer()
		{
			QueryPerformanceFrequency(&_frequence);
			_overhead = getOverhead();
		}

		//!
		//! Starts the timer
		//!
		void start()
		{
			QueryPerformanceCounter(&_start);
		}

		//!
		//! Stops the timer
		//!
		void stop()
		{
			QueryPerformanceCounter(&_stop);
		}

		//!
		//! Returns elapsed time in milliseconds (ms)
		//!
		//! \returns the elapsed time in milliseconds (ms)
		//!
		double elapsed()
		{
			return (_stop.QuadPart - _start.QuadPart - _overhead) * 1000.0 / _frequence.QuadPart;
		}

	private:

		//!
		//! Returns the overhead of the timer in ticks
		//!
		//! \returns the overhead of the timer in ticks
		//!
		LONGLONG getOverhead()
		{
			start();
			stop();
			return _stop.QuadPart - _start.QuadPart;
		}

		// members
		LARGE_INTEGER _start;
		LARGE_INTEGER _stop;
		LARGE_INTEGER _frequence;
		LONGLONG _overhead;
	};

#elif defined (__GNUC__)

	struct Timer
	{
		//!
		//! A constructor
		//!
		Timer() = default;

		//!
		//! Starts the timer
		//!
		void start()
		{
			clock_gettime(CLOCK_MONOTONIC, &_start);
		}

		//!
		//! Stops the timer
		//!
		void stop()
		{
			clock_gettime(CLOCK_MONOTONIC, &_stop);
		}

		//!
		//! Returns elapsed time in milliseconds (ms)
		//!
		//! \returns the elapsed time in milliseconds (ms)
		//!
		double elapsed()
		{
			std::chrono::nanoseconds start = std::chrono::seconds{ _start.tv_sec } + std::chrono::nanoseconds{ _start.tv_nsec };
			std::chrono::nanoseconds stop = std::chrono::seconds{ _stop.tv_sec } + std::chrono::nanoseconds{ _stop.tv_nsec };
			auto d = stop - start;
			return d.count() / 1000000.0;
		}

	private:

		// members
		struct timespec _start;
		struct timespec _stop;
	};

#endif
}
