//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <cstdio>
//---------------------------------------------------------------------------
#include <boost/bind.hpp>
#include <boost/lambda/bind.hpp>
#include <boost/foreach.hpp>
//---------------------------------------------------------------------------
#include <Wt/WBreak>
#include <Wt/WButtonGroup>
#include <Wt/WCheckBox>
#include <Wt/WFileUpload>
#include <Wt/WGroupBox>
#include <Wt/WImage>
#include <Wt/WLabel>
#include <Wt/WLineEdit>
#include <Wt/WMenu>
#include <Wt/WPushButton>
#include <Wt/WRadioButton>
#include <Wt/WStackedWidget>
#include <Wt/WTextArea>
//---------------------------------------------------------------------------
#include "administrator.h"
#include "group.h"
#include "groupfinished.h"
#include "grouploggedin.h"
#include "groupnotloggedin.h"
#include "groups.h"
#include "wtgroupwidget.h"
#include "logfile.h"
#include "parameters.h"
#include "parametersassignpayoff.h"
#include "parameterschat.h"
#include "parameterschooseaction.h"
#include "parametersfinished.h"
#include "parametersgroupdynamics.h"
#include "parametersviewresultsall.h"
#include "parametersviewresultsgroup.h"
#include "parametersviewresultsvoting.h"
#include "parametersvoting.h"
#include "stopwatch.h"
#include "administratordialog.h"
#include "administratordialogstateloggedin.h"
#include "server.h"
#include "serverstateassignpayoff.h"
#include "serverstatechat.h"
#include "serverstatechooseaction.h"
#include "serverstatefinished.h"
#include "serverstategroupdynamics.h"
#include "serverstateviewresultsall.h"
#include "serverstateviewresultsgroup.h"
#include "serverstateviewresultsvoting.h"
#include "serverstatevoting.h"
#include "serverstatewaiting.h"
#include "administratordialogstateloggedin.h"
//---------------------------------------------------------------------------
AdministratorDialogStateLoggedIn::AdministratorDialogStateLoggedIn(
  AdministratorDialog * const dialog)
  : AdministratorDialogState(dialog),
    m_button_group_set_phase(0),
    m_fileupload(0),
    m_label_state_server(0),
    m_label_state_upload(0),
    //m_text_chat(0),
    m_text_groups(0),
    m_text_parameter_file(0),
    m_text_participants(0),
    m_text_server(0)
{
  assert(!m_button_group_set_phase);
  assert(!m_fileupload);
  assert(!m_label_state_server);
  assert(!m_label_state_upload);
  //assert(!m_text_chat);
  assert(!m_text_groups);
  assert(!m_text_parameter_file);
  assert(!m_text_server);
}
//---------------------------------------------------------------------------
Wt::WContainerWidget * AdministratorDialogStateLoggedIn::CreateManipulateExperimentDialog()
{
  Wt::WContainerWidget * const dialog = new Wt::WContainerWidget;
  assert(dialog);

  new Wt::WBreak(dialog);

  //ButtonGroup
  {
    Wt::WGroupBox * const container = new Wt::WGroupBox("Experiment state",dialog);
    assert(container);
    m_button_group_set_phase = new Wt::WButtonGroup(dialog);
    assert(m_button_group_set_phase);

    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Waiting", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,0);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Group dynamics", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,1);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Chatting", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,2);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Voting", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,3);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("View voting results", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,4);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Choosing an action", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,5);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Assign individual\'s payoff", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,6);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Viewing the group\'s results", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,7);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Viewing all group\'s results", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,8);
    }
    {
      Wt::WRadioButton * const button = new Wt::WRadioButton("Finished", container);
      assert(button);
      new Wt::WBreak(container);
      m_button_group_set_phase->addButton(button,9);
    }
  }
  new Wt::WBreak(dialog);
  {
    Wt::WPushButton * const button
      = new Wt::WPushButton("Set experiment state",dialog);
      assert(button);
    button->clicked().connect(this,&AdministratorDialogStateLoggedIn::OnSetExperimentState);
  }
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  {
    Wt::WPushButton * button = new Wt::WPushButton("Delete all participants",dialog);
    button->clicked().connect(
      this,
      &AdministratorDialogStateLoggedIn::OnDeleteParticipants);
  }
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  //new Wt::WImage("TotalExperimentFlow.png",dialog);
  //new Wt::WBreak(dialog);
  //new Wt::WBreak(dialog);
  //new Wt::WBreak(dialog);
  new Wt::WImage("ExperimentFlow.png",dialog);
  //new Wt::WBreak(dialog);
  //new Wt::WBreak(dialog);
  //new Wt::WBreak(dialog);
  new Wt::WImage("IPGG.png",dialog);
  return dialog;
}
//---------------------------------------------------------------------------
Wt::WContainerWidget * AdministratorDialogStateLoggedIn::CreateStartExperimentDialog()
{
  Wt::WContainerWidget * const dialog = new Wt::WContainerWidget;
  assert(dialog);

  m_fileupload = new Wt::WFileUpload;
  m_label_state_upload = new Wt::WLabel;

  assert(m_fileupload);
  assert(m_label_state_upload);

  dialog->setContentAlignment(Wt::AlignCenter);
  new Wt::WBreak(dialog);
  dialog->addWidget(m_label_state_upload);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  dialog->addWidget(m_fileupload);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);

  m_label_state_upload->setText("Please select a file to upload or press the button");

  //Upload automatically when the user entered a file
  m_fileupload->changed().connect(
    m_fileupload,
    &Wt::WFileUpload::upload);

  //Call WtTextUploadDialog::on_upload_done when file is uploaded
  m_fileupload->uploaded().connect(
    this,
    &AdministratorDialogStateLoggedIn::OnUploadDone);
  return dialog;
}
//---------------------------------------------------------------------------
Wt::WContainerWidget * AdministratorDialogStateLoggedIn::CreateViewGroupsDialog()
{
  Wt::WContainerWidget * const dialog = new Wt::WContainerWidget;
  assert(dialog);

  m_text_groups = new Wt::WTextArea;
  m_group_widget = new WtGroupWidget;

  assert(m_text_groups);
  assert(m_group_widget);

  dialog->setContentAlignment(Wt::AlignCenter);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  dialog->addWidget(m_group_widget);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  dialog->addWidget(m_text_groups);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  new Wt::WImage("Groups.png",dialog);
  m_text_groups->setReadOnly(true);
  m_text_groups->setMinimumSize(600,200);
  m_group_widget->resize(400,400);
  return dialog;
}
//---------------------------------------------------------------------------
Wt::WContainerWidget * AdministratorDialogStateLoggedIn::CreateViewParametersDialog()
{
  Wt::WContainerWidget * const dialog = new Wt::WContainerWidget;
  assert(dialog);

  m_text_parameter_file = new Wt::WTextArea;

  assert(m_text_parameter_file);

  dialog->setContentAlignment(Wt::AlignCenter);
  new Wt::WBreak(dialog);
  m_text_parameter_file->setReadOnly(true);
  m_text_parameter_file->setMinimumSize(600,300);
  dialog->addWidget(m_text_parameter_file);

  return dialog;
}
//---------------------------------------------------------------------------
Wt::WContainerWidget * AdministratorDialogStateLoggedIn::CreateViewParticipantsDialog()
{
  Wt::WContainerWidget * const dialog = new Wt::WContainerWidget;
  assert(dialog);

  m_text_participants = new Wt::WTextArea;

  assert(m_text_participants);

  dialog->setContentAlignment(Wt::AlignCenter);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  dialog->addWidget(m_text_participants);
  m_text_participants->setReadOnly(true);
  m_text_participants->setMinimumSize(600,300);
  return dialog;
}
//---------------------------------------------------------------------------
Wt::WContainerWidget * AdministratorDialogStateLoggedIn::CreateViewServerDialog()
{
  Wt::WContainerWidget * const dialog = new Wt::WContainerWidget;
  assert(dialog);

  m_label_state_server = new Wt::WLabel;
  m_text_server = new Wt::WTextArea;

  assert(m_label_state_server);
  assert(m_text_server);

  dialog->setContentAlignment(Wt::AlignCenter);
  new Wt::WBreak(dialog);
  dialog->addWidget(m_label_state_server);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  new Wt::WLabel("The state of this text area is only changed upon button pressed,",dialog);
  new Wt::WBreak(dialog);
  new Wt::WLabel("instead of being updated automatically",dialog);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);
  dialog->addWidget(m_text_server);
  new Wt::WBreak(dialog);
  {
    Wt::WPushButton * const button
      = new Wt::WPushButton("View log file",dialog);
    assert(button);
    button->clicked().connect(
      this,&AdministratorDialogStateLoggedIn::OnViewLogFile);
  }
  {
    Wt::WPushButton * const button
      = new Wt::WPushButton("Clear log file",dialog);
    assert(button);
    button->clicked().connect(
      this,&AdministratorDialogStateLoggedIn::OnClearLogFile);
  }

  m_label_state_server->setText("Server state: ... (updated automatically)");
  m_text_server->setReadOnly(true);
  m_text_server->setMinimumSize(600,300);

  return dialog;
}
//---------------------------------------------------------------------------
const std::vector<std::vector<int> > AdministratorDialogStateLoggedIn::ExtractIds() const
{
  std::vector<std::vector<int> > v;
  //Instead of collecting the GroupUnassigned, just add an empty group to v
  v.push_back(std::vector<int>());

  BOOST_FOREACH(const Group * const group,
    Server::Get()->GetGroups()->CollectGroups(false,true,true,true))
  {
    std::vector<int> w;
    const std::vector<const Participant *> p
      = group->CollectParticipants();
    std::transform(
      p.begin(), p.end(),
      std::back_inserter(w),
      boost::bind(&Participant::GetId,boost::lambda::_1));
    assert(w.size() == p.size());
    v.push_back(w);
  }
  return v;
}
//---------------------------------------------------------------------------
void AdministratorDialogStateLoggedIn::OnDeleteParticipants()
{
  Server::Get()->DeleteParticipants();
}
//---------------------------------------------------------------------------
void AdministratorDialogStateLoggedIn::OnSetExperimentState()
{
  assert(Server::Get());

  ServerState * new_state = 0;
  switch (m_button_group_set_phase->selectedButtonIndex())
  {
    case -1: //Administarator did not click a radiobutton
      return;
    case 0: new_state = Server::Get()->GetStateWaiting(); break;
    case 1: new_state = Server::Get()->GetStateGroupDynamics(); break;
    case 2: new_state = Server::Get()->GetStateChat(); break;
    case 3: new_state = Server::Get()->GetStateVoting(); break;
    case 4: new_state = Server::Get()->GetStateViewResultsVoting(); break;
    case 5: new_state = Server::Get()->GetStateChooseAction(); break;
    case 6: new_state = Server::Get()->GetStateAssignPayoff(); break;
    case 7: new_state = Server::Get()->GetStateViewResultsGroup(); break;
    case 8: new_state = Server::Get()->GetStateViewResultsAll(); break;
    case 9: new_state = Server::Get()->GetStateFinished(); break;
  }
  assert(new_state && "New state must be known and valid");
  Server::Get()->SetState(new_state);


  assert("Should not get here");
}
//---------------------------------------------------------------------------
///Show all automatically
void AdministratorDialogStateLoggedIn::OnTimer()
{
  //std::clog << "WtAdministratorDialogStateLoggedIn::OnTimer()\n";
  //Update server state
  {
    const std::string text
      = std::string("Current server state: \'")
      + Server::Get()->GetCurrentState()->ToStr()
      + std::string("\' (updated automatically)");
    assert(m_label_state_server);
    m_label_state_server->setText(text.c_str());
  }
  //Display groups
  {
    std::string text;
    //First display the Group its Participants
    BOOST_FOREACH(
      const Group * const group,
      Server::Get()->GetGroups()->CollectGroups())
    {
      text+=std::string("Group #")
        + boost::lexical_cast<std::string>(group->GetId())
        + std::string(" (")
        + group->ToStr()
        + std::string("):\n- Participants: ");
      BOOST_FOREACH(const Participant * const participant,
        group->CollectParticipants())
      {
        text
          += (participant->CanGetId()
            ? boost::lexical_cast<std::string>(participant->GetId())
            : std::string("no ID"))
          + std::string(", ");
      }
      if (group->CollectParticipants().empty())
      {
        text+= "no participants";
      }
      else
      {
        assert(text.size() >= 2);
        //Pop trailing ', '
        text.resize(text.size() -2);
      }
      text+="\n- Payoffs: ";
      if (!group->GetAveragePayoffs().empty())
      {
        text
          +=std::string(" (")
          + boost::lexical_cast<std::string>(
            group->GetAveragePayoffs().size())
          + std::string("): ");

        BOOST_FOREACH(const double& payoff,
          group->GetAveragePayoffs())
        {
          text
            += boost::lexical_cast<std::string>(payoff)
            + std::string(", ");
        }
        text.resize(text.size() - 2);
      }
      else
      {
        text+= "no payoffs assigned";
      }
      text+="\n";
    }

    assert(!text.empty());
    m_text_groups->setText(text.c_str());

    m_group_widget->SetIds(ExtractIds());
  }
  //Display Parameters
  {

    std::stringstream s;
    s << (*Server::Get()->GetParameters());
    const std::vector<std::string> v = XmlToPretty(s.str());
    std::string text;
    BOOST_FOREACH(const std::string t,v)
    {
      text += t + '\n';
    }
    assert(!text.empty());
    text.resize(text.size() - 1);
    m_text_parameter_file->setText(text.c_str());
  }
  //Display Participants
  {
    std::string text
      = "Number of participants: "
      + boost::lexical_cast<std::string>(
        Server::Get()->GetGroups()->CollectParticipants().size())
      + '\n';
    BOOST_FOREACH(const Participant * participant,
      Server::Get()->GetGroups()->CollectParticipants())
    {
      text+=participant->ToAdminStr()+'\n';
    }
    if (text.empty())
    {
      text = "[No participants]";
    }
    else
    {
      //Pop trailing newline
      text.resize(text.size() - 1);
    }

    m_text_participants->setText(text.c_str());
  }
}
//---------------------------------------------------------------------------
void AdministratorDialogStateLoggedIn::ShowPage()
{
  AdministratorDialog * const dialog = GetDialog();
  assert(dialog);

  //dialog->StartTimer();

  dialog->clear(); //Wt deletes all dialog's child widgets

  assert(IsLoggedIn() && "Assume a logged in administrator");

  new Wt::WLabel("Welcome administrator",dialog);
  new Wt::WBreak(dialog);
  new Wt::WBreak(dialog);

  //Menu
  {
    Wt::WStackedWidget * const contents = new Wt::WStackedWidget;
    Wt::WMenu * const menu = new Wt::WMenu(contents,Wt::Horizontal,dialog);
    //Using CSS styleclass is the best (only?) way to display the menu well
    menu->setStyleClass("menu");
    {
      Wt::WMenuItem * const item = new Wt::WMenuItem(
        "Start",
        CreateStartExperimentDialog());
      menu->addItem(item);
    }
    //{
    //  Wt::WMenuItem * const item = new Wt::WMenuItem(
    //    "Chat",
    //    CreateViewChatDialog());
    //  menu->addItem(item);
    //}
    {
      Wt::WMenuItem * const item = new Wt::WMenuItem(
        "Groups",
        CreateViewGroupsDialog());
      menu->addItem(item);
    }
    {
      Wt::WMenuItem * const item = new Wt::WMenuItem(
        "Parameters",
        CreateViewParametersDialog());
      menu->addItem(item);
    }
    {
      Wt::WMenuItem * const item = new Wt::WMenuItem(
        "Participants",
        CreateViewParticipantsDialog());
      menu->addItem(item);
    }
    {
      Wt::WMenuItem * const item = new Wt::WMenuItem(
        "Server",
        CreateViewServerDialog());
      menu->addItem(item);
    }
    {
      Wt::WMenuItem * const item = new Wt::WMenuItem(
        "Manipulate",
        CreateManipulateExperimentDialog());
      menu->addItem(item);
    }
    dialog->addWidget(contents);
  }
}
//---------------------------------------------------------------------------
void AdministratorDialogStateLoggedIn::OnClearLogFile()
{
  std::remove(LogFile::m_log_filename.c_str());
  OnViewLogFile();

}
//---------------------------------------------------------------------------
void AdministratorDialogStateLoggedIn::OnUploadDone()
{  assert(FileExists(m_fileupload->spoolFileName()));
  //Display parameter file
  {
    const std::vector<std::string> v
      = FileToVector(m_fileupload->spoolFileName());
    std::string text;
    BOOST_FOREACH(const std::string& s,v)
    {
      text+=(s+'\n');
    }
    //Pop trailing newline
    text.resize(text.size() - 1);
    assert(m_text_parameter_file);
    m_text_parameter_file->setText(text.c_str());
  }

  boost::shared_ptr<Parameters> parameters(new Parameters);
  assert(parameters);
  try
  {
    assert(FileExists(m_fileupload->spoolFileName()));
    parameters->ReadFromFile(m_fileupload->spoolFileName());
  }
  catch (std::runtime_error& e)
  {
    m_label_state_upload->setText(e.what());
    return;
  }

  m_label_state_upload->setText("OK: parameter file loaded");
  Server::Get()->SetParameters(parameters);
}

//---------------------------------------------------------------------------
void AdministratorDialogStateLoggedIn::OnViewLogFile()
{
  std::string text;
  if (!FileExists(LogFile::m_log_filename))
  {
    text = "[no log file present]";
  }
  else
  {
    const std::vector<std::string> v = FileToVector(LogFile::m_log_filename);

    BOOST_FOREACH(const std::string& s,v)
    {
      //text = s + '\n' + text; //Reverse ordering
      text += s + '\n';
    }
  }
  m_text_server->setText(text.c_str());
}
//---------------------------------------------------------------------------
///Split an XML std::string into its parts
//From http://www.richelbilderbeek.nl/CppSplitXml.htm
const std::vector<std::string> AdministratorDialogStateLoggedIn::SplitXml(const std::string& s)
{
  std::vector<std::string> v;
  std::string::const_iterator i = s.begin();
  std::string::const_iterator j = s.begin();
  const std::string::const_iterator end = s.end();
  while (j!=end)
  {
    ++j;
    if ((*j=='>' || *j == '<') && std::distance(i,j) > 1)
    {
      std::string t;
      std::copy(
        *i=='<' ? i   : i+1,
        *j=='>' ? j+1 : j,
        std::back_inserter(t));
      v.push_back(t);
      i = j;
    }
  }
  return v;
}
//---------------------------------------------------------------------------
///Pretty-print an XML std::string
//From http://www.richelbilderbeek.nl/CppXmlToPretty.htm
const std::vector<std::string> AdministratorDialogStateLoggedIn::XmlToPretty(const std::string& s)
{
  std::vector<std::string> v = SplitXml(s);
  int n = -2;
  BOOST_FOREACH(std::string& s,v)
  {
    assert(!s.empty());
    if (s[0] == '<' && s[1] != '/')
    {
      n+=2;
    }
    s = std::string(n,' ') + s;
    if (s[n+0] == '<' && s[n+1] == '/')
    {
      n-=2;
    }
  }
  return v;
}
//---------------------------------------------------------------------------
