//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <cassert>
#include <cstdlib>
#include <iostream>
#include <set>
#include <stdexcept>
//---------------------------------------------------------------------------
#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/numeric/conversion/cast.hpp>
//---------------------------------------------------------------------------
#include "chooseactionoption.h"
#include "group.h"
#include "votingoption.h"
//---------------------------------------------------------------------------
Group::Group()
{

}
//---------------------------------------------------------------------------
///Add a Participant to the Group
void Group::AddParticipant(boost::shared_ptr<Participant> participant)
{
  assert(participant);
  std::clog << "Group::AddParticipant " << participant << '\n';
  this->m_participants.push_back(participant);
}
//---------------------------------------------------------------------------
///Append the last IPGG's cycle(s) average payoff to m_payoffs
void Group::AppendAveragePayoff(const double payoff)
{
  m_average_payoffs.push_back(payoff);
}
//---------------------------------------------------------------------------
///Assign the payoffs to this group
void Group::AssignPayoff(const VotingOption * const option) const
{
  assert(option);

  //Obtain the payoff contributed to the group
  double total_group_payoff = 0.0;
  BOOST_FOREACH(
    const boost::shared_ptr<Participant>& participant,m_participants)
  {
    assert(!participant->GetActions().empty());
    total_group_payoff += participant->GetActions().back()->GetContribution();
  }

  const int size = this->GetSize();
  assert(size!=0);

  std::set<boost::shared_ptr<Participant> > benefactors;
  //Collect the benefactors
  BOOST_FOREACH(
    const boost::shared_ptr<Participant>& participant,m_participants)
  {
    assert(!participant->GetActions().empty());
    double contribution = participant->GetActions().back()->GetContribution();
    if (contribution != 0.0 || GetRandomUniform() < option->GetChance())
    {
      //A contributor or a non-detected non-contributor
      benefactors.insert(participant);
    }
  }
  const double group_benefit_share
    = total_group_payoff
    / (benefactors.empty()
      ? 1.0
      : boost::numeric_cast<double>(benefactors.size()));


  {
    //Show group sizes
    //typedef std::pair<int,double> Pair;
    //BOOST_FOREACH(const Pair& p,group_payoffs)
    //{
    //  std::clog
    //    << "Group #"
    //    << p.first
    //    << " has a total contribution of "
    //    << p.second
    //    << '\n';
    //}
  }

  ///Assign each participants' payoff
  BOOST_FOREACH(
    const boost::shared_ptr<Participant>& participant,
      m_participants)
  {
    assert(!participant->GetActions().empty());
    const double individual_payoff
      = -participant->GetActions().back()->GetCost()
      + (benefactors.find(participant) != benefactors.end()
        ? group_benefit_share
        : 0.0);
    participant->AssignPayoff(individual_payoff);
    //#ifndef NDEBUG
    //std::clog
    //  <<"#"
    //  << participant->GetId()
    //  << " has contributed and gains "
    //  << individual_payoff
    //  << '\n';
    //#endif
  }
}
//---------------------------------------------------------------------------
///Checks if there is a last average payoff
//bool Group::CanGetLastAveragePayoff() const
//{
//  return !m_average_payoffs.empty();
//}
//---------------------------------------------------------------------------
///Clear all Participants
void Group::Clear()
{
  m_participants.resize(0);
}
//---------------------------------------------------------------------------
///Collects all Participants in this Group as read-only pointers
const std::vector<const Participant *> Group::CollectParticipants() const
{
  std::vector<const Participant *> v;
  BOOST_FOREACH(const boost::shared_ptr<Participant>& p,
    m_participants)
  {
    assert(p);
    v.push_back(p.get());
  }
  return v;

}
//---------------------------------------------------------------------------
///Obtain a random double in the range [0.0,1.0>
//From http://www.richelbilderbeek.nl/CppGetRandomUniform.htm
double Group::GetRandomUniform()
{
  return static_cast<double>(std::rand())/static_cast<double>(RAND_MAX);
}
//---------------------------------------------------------------------------
///Get the Group its size
int Group::GetSize() const
{
  return boost::numeric_cast<int>(m_participants.size());
}
//---------------------------------------------------------------------------
///Checks if a Participant is present in this Group
bool Group::IsMember(const Participant * const participant) const
{
  return std::find_if(
    m_participants.begin(),
    m_participants.end(),
    boost::bind(&boost::shared_ptr<Participant>::get,boost::lambda::_1)
      == participant)
  != m_participants.end();
}
//---------------------------------------------------------------------------
///Remove a Participant from a Group
boost::shared_ptr<Participant> Group::RemoveParticipant(const Participant * const participant)
{

  assert(IsMember(participant));
  BOOST_FOREACH(boost::shared_ptr<Participant>& p,m_participants)
  {
    if (participant == p.get())
    {
      boost::shared_ptr<Participant> q = p;
      std::swap(p,m_participants.back());
      m_participants.pop_back();
      return q;
    }
  }
  assert(!"Should not get here");
  throw std::logic_error("Group::RemoveParticipant called for unknown Participant");
}
//---------------------------------------------------------------------------
