//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <algorithm>
#include <cassert>
#include <iostream>
#include <numeric>
#include <stdexcept>
//---------------------------------------------------------------------------
#include <boost/foreach.hpp>
#include <boost/bind.hpp>
#include <boost/lambda/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/numeric/conversion/cast.hpp>
//#include <boost/regex.hpp>
//---------------------------------------------------------------------------
//#include "copy_if.h"
#include "groupassigner.h"
#include "groups.h"
#include "groupfinished.h"
#include "grouploggedin.h"
#include "groupnotloggedin.h"
#include "groupparticipating.h"
#include "ipaddress.h"
#include "state.h"
//---------------------------------------------------------------------------
Groups::Groups()
  : m_finished(new GroupFinished),
    m_last_id_participant(0),
    m_logged_in(new GroupLoggedIn),
    m_not_logged_in(new GroupNotLoggedIn)
{

}
//---------------------------------------------------------------------------
///Check if a Participant can be from his/her non-wildcard IP address
bool Groups::CanFind(const SafeIpAddress * const ip_address) const
{
  assert(ip_address);

  const std::vector<const Participant *> v = CollectParticipants();

  if (v.empty()) return false;

  assert(std::find_if(
      v.begin(),v.end(),
      !boost::lambda::_1) == v.end());

  const std::vector<const Participant *>::const_iterator i
    = std::find_if(
      v.begin(),v.end(),
      boost::bind(&Participant::CanGetIpAddress,boost::lambda::_1)
        && boost::bind(&SafeIpAddress::Get,
             boost::bind(&Participant::GetIpAddress,boost::lambda::_1))
            == ip_address->Get() );

  return i!=v.end();
}
//---------------------------------------------------------------------------
///Search the Group for a Participant with a certain IP address
bool Groups::CanGetParticipantWithIpAddress(const SafeIpAddress * const ip_address) const
{
  assert(ip_address);
  //Obtain all Participants
  const std::vector<const Participant *> p = this->CollectParticipants();

  return std::find_if(
    p.begin(),
    p.end(),
    boost::bind(&Participant::CanGetIpAddress,boost::lambda::_1)
      && boost::bind(&SafeIpAddress::Get,
           boost::bind(&Participant::GetIpAddress,boost::lambda::_1))
      == ip_address->Get()
    ) != p.end();
}
//---------------------------------------------------------------------------
///A Participant can log in, if
///- he/she reloaded the page by pressing F5, recovery of all actions
///- he/she started viewing the page, start of new actions
bool Groups::CanLetLogin(const SafeIpAddress * ip_address) const
{
  assert(ip_address);
  //Obtain all Participants
  const std::vector<const Participant *> p = this->CollectParticipants();

  return std::find_if(
    p.begin(),
    p.end(),
    !boost::bind(&Participant::CanGetIpAddress,boost::lambda::_1)
      || boost::bind(&SafeIpAddress::Get,
           boost::bind(&Participant::GetIpAddress,boost::lambda::_1))
      == ip_address->Get()
    ) != p.end();
}
//---------------------------------------------------------------------------
///Collect all Groups as a read-only std::vector
/*
const std::vector<const GroupParticipating *> Groups::CollectActiveGroups() const
{
  std::vector<const GroupParticipating *> v;
  BOOST_FOREACH(const boost::shared_ptr<GroupParticipating>& group,m_participating)
  {
    v.push_back(group.get());
  }
  return v;
}
*/
//---------------------------------------------------------------------------
///Collect all Participants as a read-only std::vector
/*
const std::vector<const Participant *> Groups::CollectActiveParticipants() const
{
  std::vector<const Participant *> v;
  BOOST_FOREACH(const Group * const group,CollectAllGroups())
  {
    const std::vector<const Participant *> w = group->CollectParticipants();
    std::copy(w.begin(),w.end(),std::back_inserter(v));
  }
  return v;
}
*/
//---------------------------------------------------------------------------
///Collect all Groups as a read-only std::vector
const std::vector<const Group *> Groups::CollectGroups(
  const bool not_logged_in,
  const bool logged_in,
  const bool participating,
  const bool finished) const
{
  std::vector<const Group *> v;
  if (not_logged_in)
  {
    v.push_back(m_not_logged_in.get());
  }
  if (logged_in)
  {
    v.push_back(m_logged_in.get());
  }
  if (participating)
  {
    BOOST_FOREACH(const boost::shared_ptr<GroupParticipating>& p,
      m_participating)
    {
      v.push_back(p.get());
    }
  }
  if (finished)
  {
    v.push_back(m_finished.get());
  }
  return v;
}
//---------------------------------------------------------------------------
///Collect all Participants as a read-only std::vector
const std::vector<const Participant *> Groups::CollectParticipants(
  const bool not_logged_in,
  const bool logged_in,
  const bool participating,
  const bool finished) const
{
  std::vector<const Participant *> v;
  BOOST_FOREACH(const Group * group,
    CollectGroups(not_logged_in,logged_in,participating,finished))
  {
    const std::vector<const Participant *> w = group->CollectParticipants();
    std::copy(w.begin(),w.end(),std::back_inserter(v));
  }
  /*
  #define DEBUG_27634528548479065823763297276372665273632
  #ifdef  DEBUG_27634528548479065823763297276372665273632
  if(boost::numeric_cast<int>(v.size())
    != m_finished->GetSize()
     + m_logged_in->GetSize()
     + m_not_logged_in->GetSize()
     + std::accumulate(
        m_participating.begin(),
        m_participating.end(),
        static_cast<int>(0),
        boost::bind(
         std::plus<int>(),_1,
          boost::bind(&Group::GetSize,boost::lambda::_2))))
  {
    std::clog
      << "not_logged_in: " << not_logged_in << '\n'
      << "logged_in: " << logged_in << '\n'
      << "participating: " << participating << '\n'
      << "finished: " << finished << '\n'
      << "v: " << v.size() << '\n'
      << "m_finished:" << m_finished->GetSize() << '\n'
      << "m_logged_in: " << m_logged_in->GetSize() << '\n'
      << "m_not_logged_in: " << m_not_logged_in->GetSize() << '\n'
      << "others: " << std::accumulate(
        m_participating.begin(),
        m_participating.end(),
        static_cast<int>(0),
        boost::bind(
         std::plus<int>(),_1,
          boost::bind(&Group::GetSize,boost::lambda::_2))) << '\n';
  }
  #endif
  assert(boost::numeric_cast<int>(v.size())
    == m_finished->GetSize()
     + m_logged_in->GetSize()
     + m_not_logged_in->GetSize()
     + std::accumulate(
        m_participating.begin(),
        m_participating.end(),
        static_cast<int>(0),
        boost::bind(
         std::plus<int>(),_1,
          boost::bind(&Group::GetSize,boost::lambda::_2))));
  */
  return v;
}
//---------------------------------------------------------------------------
///Collect the Group with unassigned Participants as a read-only Group
/*
const std::vector<const Group *> Groups::CollectPassiveGroups() const
{
  std::vector<const Group *> v;
  v.push_back(m_finished.get());
  v.push_back(m_logged_in.get());
  v.push_back(m_not_logged_in.get());
  return v;
}
*/
//---------------------------------------------------------------------------
/*
///Collect all unassigned Participants as a read-only std::vector
const std::vector<const Participant *> Groups::CollectPassiveParticipants() const
{
  std::vector<const Participant *> p;
  std::vector<const Group *> groups = CollectPassiveGroups();
  BOOST_FOREACH(const Group* const group,groups)
  {
    const std::vector<const Participant *> q
      = group->CollectParticipants();
    std::copy(q.begin(),q.end(),std::back_inserter(p));
  }
  return p;
}
*/
//---------------------------------------------------------------------------
///Find a Participant from his/her non-wildcard IP address
const Participant * Groups::Find(const SafeIpAddress * const ip_address) const
{
  assert(ip_address);
  assert(CanFind(ip_address));

  const std::vector<const Participant *> v = CollectParticipants();

  assert(std::find_if(
      v.begin(),v.end(),
      !boost::lambda::_1) == v.end());

  //Search for a Participant with this IP address
  {
    const std::vector<const Participant *>::const_iterator i
      = std::find_if(
        v.begin(),v.end(),
        boost::bind(&Participant::CanGetIpAddress,boost::lambda::_1)
          && boost::bind(&SafeIpAddress::Get,
               boost::bind(&Participant::GetIpAddress,boost::lambda::_1))
              == ip_address->Get() );

    //A Participant with a known IP address is found
    if (i!=v.end())
    {
      assert(*i);
      assert((*i)->CanGetIpAddress());
      assert((*i)->GetIpAddress()->Get() == ip_address->Get());
      return *i;
    }
  }

  //Search for a Participant without an IP address assiged
  /*
  {
    const std::vector<const Participant *>::const_iterator i
      = std::find_if(
        v.begin(),v.end(),
        !boost::bind(&Participant::CanGetIpAddress,boost::lambda::_1));

    if (i!=v.end())
    {
      assert(*i);
      assert(!(*i)->CanGetIpAddress());
      return *i;
    }
  }
  */
  assert(!"Should not get here");
  return 0;
}
//---------------------------------------------------------------------------
///Find a read/write Group from a read-only Group
///Just checks if the Group really exists
///
///\note const_cast!
Group * Groups::FindGroup(const Group * const group) const
{
  assert(group);

  #ifndef NDEBUG
  const std::vector<const Group *> groups = CollectGroups();
  assert(std::find_if(
    groups.begin(),groups.end(),
    boost::lambda::_1 == group) != groups.end());
  #endif

  return const_cast<Group*>(group);
}
//---------------------------------------------------------------------------
///Finds the Group the Participant is in
const Group * Groups::FindMyGroup(const Participant * const participant) const
{
  BOOST_FOREACH(const Group * const group,CollectGroups())
  {
    if (group->IsMember(participant))
    return group;
  }
  assert(!"All Participants must be in a Group");
  return 0;
}
//---------------------------------------------------------------------------
///Find a read/write Participant from a read-only Participant
///Just checks if the Participant really exists
///
///\note const_cast!
Participant * Groups::FindParticipant(const Participant * const participant) const
{
  assert(participant);
  assert(FindMyGroup(participant)!=0
    && "Every Participant must be in a Group");

  return const_cast<Participant*>(participant);
}
//---------------------------------------------------------------------------
///Find a Participant with this or a wildcard IP address
//Must return a boost::shared_ptr<Participant>
/*
boost::shared_ptr<Participant> Groups::FindParticipantByIpAddress(
  const SafeIpAddress& ip_address) const
{
  assert(CanLoginParticipant(ip_address));

  BOOST_FOREACH(boost::shared_ptr<Participant> participant,
    m_not_logged_in->GetParticipants())
  {
    assert( (!participant->CanGetIpAddress()
      || participant->GetIpAddress() != ip_address)
      && "Either a Participant has no IP address or is already logged in");
    if (!participant->HasIpAddress()) return participant;
  }
  assert(!"Should not get here");
  throw std::logic_error("Server::FindParticipant");
}
*/
//---------------------------------------------------------------------------
///Obtain the GroupLoggedIn as a read-only pointer
const GroupLoggedIn * Groups::GetGroupLoggedIn() const
{
  assert(m_logged_in);
  return m_logged_in.get();
}
//---------------------------------------------------------------------------
///Obtain the GroupNotLoggedIn as a read-only pointer
const GroupNotLoggedIn * Groups::GetGroupNotLoggedIn() const
{
  assert(m_not_logged_in);
  return m_not_logged_in.get();
}
//---------------------------------------------------------------------------
///Find the Group for the Participant with a certain IP address
const Participant * Groups::GetParticipantWithIpAddress(
  const SafeIpAddress * const ip_address) const
{
  assert(CanGetParticipantWithIpAddress(ip_address));
  const std::vector<const Participant *> p = this->CollectParticipants();

  return *std::find_if(
    p.begin(),
    p.end(),
    boost::bind(&SafeIpAddress::Get,
      boost::bind(&Participant::GetIpAddress,boost::lambda::_1))
      == ip_address->Get()
    );
}
//---------------------------------------------------------------------------
///Let a Group grow from 3 to 5 Participants
void Groups::GrowGroup(const Group * const group)
{
  GroupLoggedIn * const from_group = this->m_logged_in.get();
  Group * const to_group = FindGroup(group);
  assert(to_group->GetSize() == 3);
  if (from_group->GetSize() != 0)
  {
    to_group->AddParticipant(
      from_group->RemoveParticipant(
        from_group->CollectParticipants().back()));
  }
  if (from_group->GetSize() != 0)
  {
    to_group->AddParticipant(
      from_group->RemoveParticipant(
        from_group->CollectParticipants().back()));
  }
}
//---------------------------------------------------------------------------
///Relocate a GroupParticipating to GroupFinished
void Groups::KillGroup(const Group * const group_to_move)
{
  BOOST_FOREACH(boost::shared_ptr<GroupParticipating> group,
    m_participating)
  {
    if (group.get() == group_to_move)
    {
      const std::vector<const Participant *> v
        = group->CollectParticipants();
      BOOST_FOREACH(const Participant * p,v)
      {
        m_finished->AddParticipant(
          group->RemoveParticipant(p));
      }
      m_participating.erase(group);
      return;
    }
  }

  assert(!"Should not get here");

}
//---------------------------------------------------------------------------
///Move a Participant from the any Group to the GroupLoggedIn
const Participant * Groups::LetLogin(const SafeIpAddress * ip_address)
{
  assert(CanLetLogin(ip_address));

  ///Assign this IP address to a Participant
  if (CanFind(ip_address))
  {
    assert(Find(ip_address)->CanGetId());
    assert(Find(ip_address)->CanGetIpAddress());
    return Find(ip_address);
  }

  const std::vector<const Participant *> v = CollectParticipants();

  const Participant * const participant
    = *std::find_if(
      v.begin(),v.end(),
      !boost::bind(&Participant::CanGetIpAddress,boost::lambda::_1));
  assert(!participant->CanGetIpAddress());


  //Assign the ID
  ///\note const_cast
  const_cast<Participant*>(participant)->AssignId(++m_last_id_participant);

  assert(participant->CanGetId()
    && "Assume that the new Participant now has his/her ID assigned");

  //Assign the IP address
  ///\note const_cast
  const_cast<Participant*>(participant)->SetIpAddress(ip_address);

  assert(participant->CanGetIpAddress()
    && "Every participant must have a valid IP address after logging in");

  //Move the Participant
  assert(FindMyGroup(participant) == m_not_logged_in.get());
  assert(GetGroupNotLoggedIn()->IsMember(participant));
  assert(!GetGroupLoggedIn()->IsMember(participant));

  m_logged_in->AddParticipant(m_not_logged_in->RemoveParticipant(participant));

  assert(!GetGroupNotLoggedIn()->IsMember(participant));
  assert(GetGroupLoggedIn()->IsMember(participant));

  //Set the new state
  ///\note const_cast
  const_cast<Participant*>(participant)->SetState(new StateLoggedIn);

  return participant;
}
//---------------------------------------------------------------------------
///Split a Group of 5 Participants to 2 groups of 3
void Groups::SplitGroup(const Group * const group)
{
  assert(group->GetSize() == 5);
  assert(m_logged_in->GetSize());
  Group * group1 = FindGroup(group);
  group1->AddParticipant(
    m_logged_in->RemoveParticipant(
      m_logged_in->CollectParticipants().back()));
  assert(group1->GetSize() == 6);
  std::vector<const Participant*> p = group1->CollectParticipants();
  std::random_shuffle(p.begin(),p.end());

  boost::shared_ptr<GroupParticipating> group2(new GroupParticipating);

  for (int i=0; i!=3; ++i)
  {
    group2->AddParticipant(
      group1->RemoveParticipant(
        p[i]));
  }
  assert(group1->GetSize() == 3);
  assert(group2->GetSize() == 3);
  m_participating.insert(group2);
  //m_participating.push_back(group2);
}
//---------------------------------------------------------------------------
///MoveAllToFinished moves all Participants to the Finished Group
void Groups::MoveAllToFinished()
{
  {
    const std::vector<const Participant *> v
      = m_not_logged_in->CollectParticipants();
    BOOST_FOREACH(const Participant * p,v)
    {
      m_finished->AddParticipant(
        m_not_logged_in->RemoveParticipant(p));
    }
  }
  {
    const std::vector<const Participant *> v
      = m_logged_in->CollectParticipants();
    BOOST_FOREACH(const Participant * p,v)
    {
      m_finished->AddParticipant(
        m_logged_in->RemoveParticipant(p));
    }
  }
  BOOST_FOREACH(boost::shared_ptr<GroupParticipating> group,
    m_participating)
  {
    const std::vector<const Participant *> v
      = group->CollectParticipants();
    BOOST_FOREACH(const Participant * p,v)
    {
      m_finished->AddParticipant(
        group->RemoveParticipant(p));
    }
  }
}
//---------------------------------------------------------------------------
///Moves all logged in Participants to the GroupPartipating
void Groups::MoveLoggedInToParticipating()
{
  const std::vector<const Participant *> v
    = m_logged_in->CollectParticipants();
  BOOST_FOREACH(const Participant * p,v)
  {
    if (const GroupAssignerPredetermined * const a
      = dynamic_cast<const GroupAssignerPredetermined*>(p->GetGroupAssigner()))
    {
      const int group_index = a->GetGroup();
      //If there ain't a group with this index, create it
      while (std::find_if(m_participating.begin(),
        m_participating.end(),
        boost::bind(&Group::GetId,boost::lambda::_1)
          == group_index) == m_participating.end())
      {
        boost::shared_ptr<GroupParticipating> new_group(new GroupParticipating);
        m_participating.insert(new_group);
      }

      //Assume the group with this index can be found
      assert(std::find_if(m_participating.begin(),
        m_participating.end(),
        boost::bind(&Group::GetId,boost::lambda::_1)
          == group_index) != m_participating.end());


      //Retrieve the group with this index
      (*std::find_if(m_participating.begin(),
        m_participating.end(),
        boost::bind(&Group::GetId,boost::lambda::_1)
          == group_index))->AddParticipant(
          m_logged_in->RemoveParticipant(p));

      #ifdef USE_VECTOR_TO_STORE_PARTICIPANTS_82346234283
      while (group_index >= boost::numeric_cast<int>(m_participating.size()))
      {
        boost::shared_ptr<GroupParticipating> new_group(new GroupParticipating);
        m_participating.push_back(new_group);
      }
      m_participating[group_index]->AddParticipant(
        m_logged_in->RemoveParticipant(p));
      #endif
    }
  }
}
//---------------------------------------------------------------------------
///Remove all Participants
void Groups::Reset()
{
  m_finished->Clear();
  m_last_id_participant = 0;
  m_logged_in->Clear();
  m_not_logged_in->Clear();
  //m_participating.resize(0);
  m_participating.clear();
  GroupParticipating::Reset();
}
//---------------------------------------------------------------------------
///Set the Participants for the coming experiment
void Groups::SetParticipants(std::vector<boost::shared_ptr<Participant> > participants)
{
  Reset();
  assert(m_not_logged_in->GetSize() == 0);
  BOOST_FOREACH(const boost::shared_ptr<Participant>& p,participants)
  {
    m_not_logged_in->AddParticipant(p);
  }
}
//---------------------------------------------------------------------------
