//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#ifndef GROUPS_H
#define GROUPS_H
//---------------------------------------------------------------------------
#include <set>
#include <vector>
//---------------------------------------------------------------------------
#include <boost/checked_delete.hpp>
#include <boost/noncopyable.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/shared_ptr.hpp>
//---------------------------------------------------------------------------
#include "forward_declarations.h"
//---------------------------------------------------------------------------
///\brief
///Groups is a Facade for managing all Group instances
///
///Groups is befriended to ServerStateGroupDynamics for modifying
///the Participant Group distribution
struct Groups : public boost::noncopyable
{
  ///Groups constructor
  Groups();

  ///Check if a Participant can be from his/her non-wildcard IP address
  bool CanFind(const SafeIpAddress * const ip_address) const;

  ///Search the Group for a Participant with a certain IP address
  bool CanGetParticipantWithIpAddress(const SafeIpAddress * const ip_address) const;

  ///A Participant can log in, if
  ///- he/she reloaded the page by pressing F5, recovery of all actions
  ///- he/she started viewing the page, start of new actions
  bool CanLetLogin(const SafeIpAddress * ip_address) const;

  ///Collect all desired Groups as a read-only std::vector
  const std::vector<const Group *> CollectGroups(
    const bool not_logged_in = true,
    const bool logged_in = true,
    const bool participating = true,
    const bool finished = true) const;

  ///Collect all Participants as a read-only std::vector
  const std::vector<const Participant *> CollectParticipants(
    const bool not_logged_in = true,
    const bool logged_in = true,
    const bool participating = true,
    const bool finished = true) const;

  ///Collect all Participants that are active as a read-only std::vector
  //const std::vector<const Participant *> CollectActiveParticipants() const;

  ///Collect all Groups of active Participants as a read-only std::vector
  //const std::vector<const GroupParticipating *> CollectActiveGroups() const;

  ///Collect the Group with unassigned Participants as a read-only Group
  //const std::vector<const Group *> CollectPassiveGroups() const;

  ///Collect all unassigned Participants as a read-only std::vector
  //const std::vector<const Participant *> CollectPassiveParticipants() const;

  ///Find a Participant from his/her non-wildcard IP address
  const Participant * Find(const SafeIpAddress * const ip_address) const;

  ///Finds the Group the Participant is in
  const Group * FindMyGroup(const Participant * const participant) const;

  ///Obtain the GroupLoggedIn as a read-only pointer
  const GroupLoggedIn * GetGroupLoggedIn() const;

  ///Obtain the GroupNotLoggedIn as a read-only pointer
  const GroupNotLoggedIn * GetGroupNotLoggedIn() const;

  ///Find the Group for the Participant with a certain IP address
  const Participant * GetParticipantWithIpAddress(
    const SafeIpAddress * const ip_address) const;

  ///Move a Participant from the passive to the active Group
  const Participant * LetLogin(const SafeIpAddress * ip_address);

  ///Moves all Participants to the GroupFinished
  void MoveAllToFinished();

  ///Moves all logged in Participants to the GroupPartipating
  void MoveLoggedInToParticipating();

  ///Set the Participants for the coming experiment
  void SetParticipants(std::vector<boost::shared_ptr<Participant> > participants);

  private:
  ///Only allow a Boost smart pointer to delete Groups
  //to prevent the following trouble,
  //cited from http://www.boost.org/libs/utility/checked_delete.html:
  //The C++ Standard allows, in 5.3.5/5, pointers to incomplete
  //class types to be deleted with a delete-expression.
  //When the class has a non-trivial destructor, or a class-specific operator
  //delete, the behavior is undefined. Some compilers issue a warning when an
  //incomplete type is deleted, but unfortunately, not all do, and programmers
  //sometimes ignore or disable warnings.
  ~Groups() {}
  ///Only allow a Boost smart pointer to delete Groups
  //Template syntax from Herb Sutter. Exceptional C++ style. 2005. ISBN: 0-201-76042-8. Item 8: 'Befriending templates'.
  friend void boost::checked_delete<>(Groups*);

  ///ServerState can modify
  ///the Group distribution
  friend class ServerState;

  ///ServerStateGroupNotParticipatingGroupDynamics can modify
  ///the Participant Group distribution
  friend class ServerStateGroupDynamics;

  ///ServerStateFinished can modify
  ///the Participant Group distribution
  ///friend class ServerStateFinished;

  ///The group with Participants that are finished
  boost::scoped_ptr<GroupFinished> m_finished;

  ///The ID given to the last Participant
  int m_last_id_participant;

  ///The group with Participants that are logged in,
  ///but not assigned to a group
  boost::scoped_ptr<GroupLoggedIn> m_logged_in;

  ///The group with Participants that are not logged in
  boost::scoped_ptr<GroupNotLoggedIn> m_not_logged_in;

  ///The Participants assigned to a group
  std::set<boost::shared_ptr<GroupParticipating> > m_participating;

  ///Find a read/write Group from a read-only Group
  ///Just checks if the Group really exists
  Group * FindGroup(const Group * const group) const;

  ///Find a read/write Participant from a read-only Participant
  ///Just checks if the Participant really exists
  Participant * FindParticipant(const Participant * const participant) const;

  ///Find a Participant with this or a wildcard IP address
  //boost::shared_ptr<Participant> FindParticipantByIpAddress(const std::string& ip_address) const;

  ///Let a Group grow from 3 to 5 Participants
  void GrowGroup(const Group * const group);

  ///Remove a Group
  void KillGroup(const Group * const group);

  ///Split a Group of 6 Participants to 2 groups of 3
  void SplitGroup(const Group * const group);

  ///Remove all Participants
  void Reset();
};
//---------------------------------------------------------------------------
#endif // GROUPS_H
