//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#ifndef LOGFILE_H
#define LOGFILE_H
//---------------------------------------------------------------------------
#include <ctime>
#include <string>
#include <vector>
//---------------------------------------------------------------------------
#include <boost/shared_ptr.hpp>
#include <boost/noncopyable.hpp>
//---------------------------------------------------------------------------
#include "forward_declarations.h"
//---------------------------------------------------------------------------
///\brief
///LogFile is the log file.
///
///A crashed session should be able to be restored it.
struct LogFile : public boost::noncopyable
{
  LogFile();

  ///Clear the LogFile
  static void ClearLogFile();

  ///Log a Participant his/her last action
  void LogAction(const Participant * const participant);

  ///Log a Participant being assigned to a group
  void LogAssignGroup(
    const Group * const group,
    const Participant * const participant);

  ///Log a ChatMessage from a Participant
  void LogChatMessage(
    const Participant * const participant,
    const std::string& message);

  ///Log that the experiment its state/phase has been changed
  void LogExperimentStateChanged(const ServerState* const new_state);

  ///Log that a Participant has logged in
  void Login(const Participant * const participant);

  ///Log a Participant has logged out (although he/she will remain in memory)
  void Logout(const Participant * const participant);

  ///Log the parameters
  void LogParameters(boost::shared_ptr<Parameters> parameters);

  ///Log that a ParticipantDialog has its ParticipantDialogState changed
  void LogParticipantDialogStateChanged(const ParticipantDialogState* const new_state);

  ///Log a Participant his/her vote
  void LogVote(const Participant * const participant);

  ///The filename of the log file where all logged operations are written to
  static const std::string m_log_filename;

  private:
  ///Only allow a Boost smart pointer to delete LogFile
  //to prevent the following trouble,
  //cited from http://www.boost.org/libs/utility/checked_delete.html:
  //The C++ Standard allows, in 5.3.5/5, pointers to incomplete
  //class types to be deleted with a delete-expression.
  //When the class has a non-trivial destructor, or a class-specific operator
  //delete, the behavior is undefined. Some compilers issue a warning when an
  //incomplete type is deleted, but unfortunately, not all do, and programmers
  //sometimes ignore or disable warnings.
  ~LogFile() {}
  ///Only allow a Boost smart pointer to delete LogFile
  //Template syntax from Herb Sutter. Exceptional C++ style. 2005. ISBN: 0-201-76042-8. Item 8: 'Befriending templates'.
  friend void boost::checked_delete<>(LogFile*);

  ///Helper functions
  //static bool FileExists(const std::string& filename);
  //static const std::vector<std::string> FileToVector(const std::string& filename);

  ///Get the current time as a std::string
  //From http://www.richelbilderbeek.nl/CppGetTime.htm
  static const std::string GetTime();

  ///Get the current time as a std::time_t
  //From http://www.richelbilderbeek.nl/CppGetTime.htm
  static std::time_t GetTimeT();

  ///TimeToStr converts std::time_t to std::string.
  //From http://www.richelbilderbeek.nl/CppTimeToStr.htm
  static const std::string TimeToStr(const std::time_t& time);
};
//---------------------------------------------------------------------------
#endif // LOGFILE_H
