//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <cassert>
#include <fstream>
#include <iostream>
#include <stdexcept>
//---------------------------------------------------------------------------
#include <boost/foreach.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/numeric/conversion/cast.hpp>
#include <boost/algorithm/string/trim.hpp>
//#include <boost/regex.hpp>
//---------------------------------------------------------------------------
#include "groupassigner.h"
#include "ipaddress.h"
#include "parameters.h"
#include "parametersassignpayoff.h"
#include "parameterschat.h"
#include "parameterschooseaction.h"
#include "parametersfinished.h"
#include "parametersgroupdynamics.h"
#include "parametersviewresultsall.h"
#include "parametersviewresultsgroup.h"
#include "parametersviewresultsvoting.h"
#include "parametersvoting.h"
#include "participant.h"
//---------------------------------------------------------------------------
Parameters::Parameters()
  : //m_participants(CreateDefaultParticipants()),
    m_assign_payoff(new ParametersAssignPayoff),
    m_chat(new ParametersChat),
    m_choose_action(new ParametersChooseAction),
    m_finished(new ParametersFinished),
    m_group_dynamics(new ParametersGroupDynamics),
    m_view_results_all(new ParametersViewResultsAll),
    m_view_results_group(new ParametersViewResultsGroup),
    m_view_results_voting(new ParametersViewResultsVoting),
    m_voting(new ParametersVoting)
{
  assert(m_assign_payoff);
  assert(m_chat);
  assert(m_choose_action);
  assert(m_finished);
  assert(m_group_dynamics);
  assert(m_participants.empty()
    && "Do not create Participants by default as this will start a default experiment");
  assert(m_view_results_all);
  assert(m_view_results_group);
  assert(m_view_results_voting);
  assert(m_voting);

}
//---------------------------------------------------------------------------
///Add Participants to Parameters
void Parameters::AddParticipant(boost::shared_ptr<Participant> participant)
{
  std::clog << "Added participant " << participant << '\n';
  assert(participant);
  m_participants.push_back(participant);
}
//---------------------------------------------------------------------------
///Create a default non-logged in Partipant
boost::shared_ptr<Participant> Parameters::CreateDefaultParticipant()
{
  static int already_created = 0;
  //Create chat tag
  const char chat_tag = '!' + already_created;
  //Create GroupAssigner
  boost::shared_ptr<GroupAssigner> group_assigner(
    new GroupAssignerPredetermined(
      1 + (already_created%2)));
  //Set wild-card IP address
  const std::string ip_address="*";
  //Create the Participant
  boost::shared_ptr<Participant> p(
    new Participant(
      chat_tag,
      group_assigner));

  assert(p);
  //Check if the Participant is really created from scratch
  assert(!p->CanGetId()
    && "ID must be assigned by Server");

  assert(!p->CanGetIpAddress() && "IP address must be assigned by Server");

  ++already_created;

  return p;
}
//---------------------------------------------------------------------------
///Deletes all Participant instances
void Parameters::DeleteParticipants()
{
  this->m_participants.clear();
  assert(m_participants.empty());
}
//---------------------------------------------------------------------------
void Parameters::ReadFromFile(const std::string& filename)
{
  if (!FileExists(filename))
  {
    const std::string error
      = std::string("File ")
      + filename
      + std::string(" not found");
    throw std::runtime_error(error);
  }
  assert(FileExists(filename));

  const std::vector<std::string> v = FileToVector(filename);
  BOOST_FOREACH(const std::string& s, v)
  {
    //Trim_copy s to remove Microsoft Windows line endings
    Parse(boost::algorithm::trim_copy(s));
  }
}
//---------------------------------------------------------------------------
///FileExists checks if a certain file exists
///From http://www.richelbilderbeek.nl/CppFileExists.htm
bool Parameters::FileExists(const std::string& filename)
{
  std::fstream f;
  f.open(filename.c_str(),std::ios::in);
  return f.is_open();
}
//---------------------------------------------------------------------------
///FileToVector reads a file and converts it to a std::vector<std::string>
///From http://www.richelbilderbeek.nl/CppFileToVector.htm
const std::vector<std::string> Parameters::FileToVector(const std::string& filename)
{
  assert(FileExists(filename));
  std::vector<std::string> v;
  std::ifstream in(filename.c_str());
  std::string s;
  for (int i=0; !in.eof(); ++i)
  {
    std::getline(in,s);
    v.push_back(s);
  }
  return v;
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersAssignPayoff * Parameters::GetAssignPayoff()
{
  return const_cast<ParametersAssignPayoff*>(
    const_cast<const Parameters&>(*this).GetAssignPayoff());
}
//---------------------------------------------------------------------------
const ParametersAssignPayoff * Parameters::GetAssignPayoff() const
{
  assert(m_assign_payoff);
  return m_assign_payoff.get();
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersChat * Parameters::GetChat()
{
  return const_cast<ParametersChat*>(
    const_cast<const Parameters&>(*this).GetChat());
}
//---------------------------------------------------------------------------
const ParametersChat * Parameters::GetChat() const
{
  assert(m_chat);
  return m_chat.get();
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersChooseAction * Parameters::GetChooseAction()
{
  return const_cast<ParametersChooseAction*>(
    const_cast<const Parameters&>(*this).GetChooseAction());
}
//---------------------------------------------------------------------------
const ParametersChooseAction * Parameters::GetChooseAction() const
{
  assert(m_choose_action);
  return m_choose_action.get();
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersFinished * Parameters::GetFinished()
{
  return const_cast<ParametersFinished*>(
    const_cast<const Parameters&>(*this).GetFinished());
}
//---------------------------------------------------------------------------
const ParametersFinished * Parameters::GetFinished() const
{
  assert(m_finished);
  return m_finished.get();
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersGroupDynamics * Parameters::GetGroupDynamics()
{
  return const_cast<ParametersGroupDynamics*>(
    const_cast<const Parameters&>(*this).GetGroupDynamics());
}
//---------------------------------------------------------------------------
const ParametersGroupDynamics * Parameters::GetGroupDynamics() const
{
  assert(m_group_dynamics);
  return m_group_dynamics.get();
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersViewResultsAll * Parameters::GetViewResultsAll()
{
  return const_cast<ParametersViewResultsAll*>(
    const_cast<const Parameters&>(*this).GetViewResultsAll());
}
//---------------------------------------------------------------------------
const ParametersViewResultsAll * Parameters::GetViewResultsAll() const
{
  assert(m_view_results_all);
  return m_view_results_all.get();
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersViewResultsGroup * Parameters::GetViewResultsGroup()
{
  return const_cast<ParametersViewResultsGroup*>(
    const_cast<const Parameters&>(*this).GetViewResultsGroup());
}
//---------------------------------------------------------------------------
const ParametersViewResultsGroup * Parameters::GetViewResultsGroup() const
{
  assert(m_view_results_group);
  return m_view_results_group.get();
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersViewResultsVoting * Parameters::GetViewResultsVoting()
{
  return const_cast<ParametersViewResultsVoting*>(
    const_cast<const Parameters&>(*this).GetViewResultsVoting());
}
//---------------------------------------------------------------------------

const ParametersViewResultsVoting * Parameters::GetViewResultsVoting() const
{
  assert(m_view_results_voting);
  return m_view_results_voting.get();
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParametersVoting * Parameters::GetVoting()
{
  return const_cast<ParametersVoting*>(
    const_cast<const Parameters&>(*this).GetVoting());
}
//---------------------------------------------------------------------------
const ParametersVoting * Parameters::GetVoting() const
{
  assert(m_voting);
  return m_voting.get();
}
//---------------------------------------------------------------------------
///Parse a line in a Parameter file.
void Parameters::Parse(const std::string& s)
{
  ///Parameters starting with finished_ are parsed by ParametersFinished
  if (s.size() > 9 && s.substr(0,9) == "finished_")
  {
    const std::string t = s.substr(9,s.size()-9);
    this->GetFinished()->Parse(t);
  }

  ///Parameters starting with finished_ are parsed by ParametersGroupDynamics
  if (s.size() > 15 && s.substr(0,15) == "group_dynamics_")
  {
    const std::string t = s.substr(15,s.size()-15);
    this->GetGroupDynamics()->Parse(t);
  }

  ///Parameters starting with finished_ are parsed by ParametersGroupDynamics
  if (s.size() > 14 && s.substr(0,14) == "assign_payoff_")
  {
    const std::string t = s.substr(14,s.size()-14);
    this->GetAssignPayoff()->Parse(t);
  }

  ///Parameters starting with finished_ are parsed by ParametersViewResultsGroup
  if (s.size() > 20 && s.substr(0,20) == "view_results_voting_")
  {
    const std::string t = s.substr(20,s.size()-20);
    this->GetViewResultsVoting()->Parse(t);
  }

  ///Parameters starting with finished_ are parsed by ParametersViewResultsGroup
  if (s.size() > 19 && s.substr(0,19) == "view_results_group_")
  {
    const std::string t = s.substr(19,s.size()-19);
    this->GetViewResultsGroup()->Parse(t);
  }

  ///Parameters starting with finished_ are parsed by ParametersViewResultsGroup
  if (s.size() > 17 && s.substr(0,17) == "view_results_all_")
  {
    const std::string t = s.substr(17,s.size()-17);
    this->GetViewResultsAll()->Parse(t);
  }

  ///Parameters starting with finished_ are parsed by ParametersViewResultsGroup
  if (s.size() > 14 && s.substr(0,14) == "assign_payoff_")
  {
    const std::string t = s.substr(14,s.size()-14);
    this->GetAssignPayoff()->Parse(t);
  }

  if (s.size() > 14 && s.substr(0,14) == "choose_action_")
  {
    const std::string t = s.substr(14,s.size()-14);
    this->GetChooseAction()->Parse(t);
  }

  if (s.size() > 5 && s.substr(0,5) == "chat_")
  {
    const std::string t = s.substr(5,s.size()-5);
    this->GetChat()->Parse(t);
  }

  if (s.size() > 7 && s.substr(0,7) == "voting_")
  {
    const std::string t = s.substr(7,s.size()-7);
    this->GetVoting()->Parse(t);
  }

  //Participants




  //Participants
  if (s.size() > 12 && s.substr(0,12) == "participant=")
  {
    const std::string t = s.substr(12,s.size() - 12);
    const std::vector<std::string> v = SeperateString(t,',');
    if (v.size() != 3)
    {
      std::clog
        << "Incorrectly formed participant line: " << s << '\n'
        << "Must consist of [group assigner index][chat_tag][IP address]\n"
        << "For example '1,@,*\n";
      return;
    }
    assert(v.size() == 3 && "Participant must have three elements");

    boost::shared_ptr<GroupAssigner> group_assigner
      = GroupAssigner::CreateAssigner(v[0]);

    assert(v[1].size() == 1 && "A chat tag consists of one single character");
    const char chat_tag = v[1][0];

    const std::string ip_address_str = v[2];


    boost::shared_ptr<Participant> participant(
     new Participant(
       chat_tag,
       group_assigner)
      );
    if (ip_address_str!="*")
    {
      try
      {
        boost::scoped_ptr<SafeIpAddress> ip_address(new SafeIpAddress(ip_address_str));
        participant->SetIpAddress(ip_address.get());
      }
      catch (std::logic_error&)
      {
        std::clog
          << "Incorrectly formed IP address: "
          << ip_address_str
          << '\n';
        return;
      }
    }

    m_participants.push_back(participant);
    return;
  }
}
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/CppSeperateString.htm
const std::vector<std::string> Parameters::SeperateString(
  const std::string& input,
  const char seperator)
{
  std::istringstream is(input);
  std::vector<std::string> v;
  for (
    std::string sub;
    std::getline(is, sub, seperator);
    v.push_back(sub))
  {
    //Empty for loop
  }
  return v;
}
//---------------------------------------------------------------------------
///StealParticipant returns an unassigned participant.
///The Participant is removed from Parameters by doing this.
/*
boost::shared_ptr<Participant> Parameters::StealParticipant(const std::string& ip_address)
{
  assert(HasUnassignedParticipants()
    && "There must be at least one unassigned participant to steal (that) one");

  assert(HasUnassignedParticipant(ip_address)
    && "There must be a valid participant to steal");

  //Check if there is a participant with this IP address
  const int sz = boost::numeric_cast<int>(m_participants.size());
  for (int i=0; i!=sz; ++i)
  {
    if (m_participants[i]->GetIpAddress() == ip_address)
    {
      //Found!
      boost::shared_ptr<Participant> participant = m_participants[i];
      assert(!participant->CanGetGroup()
        && "A Participant can only be assigned to a group (after stealing) by the Server");

      //Put the found participant in the back of m_participants
      std::swap(m_participants[i],m_participants.back());

      //Remove the found participant from m_participants
      m_participants.pop_back();

      //Return the isolated participant
      return participant;
    }
  }

  //Check if there is a participant with this IP address
  assert(sz == boost::numeric_cast<int>(m_participants.size()));
  for (int i=0; i!=sz; ++i)
  {
    if (m_participants[i]->GetIpAddress() == "*")
    {
      //Found!
      boost::shared_ptr<Participant> participant = m_participants[i];
      assert(!participant->CanGetGroup()
        && "A Participant can only be assigned to a group (after stealing) by the Server");

      //Put the found participant in the back of m_participants
      std::swap(m_participants[i],m_participants.back());

      //Remove the found participant from m_participants
      m_participants.pop_back();

      //Return the isolated participant
      return participant;
    }
  }
  assert("Should not get here");
  throw std::logic_error("Parameters::StealParticipant");
}
*/
//---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& os,const Parameters& parameters)
{
  os
    << "<parameters>"
    << "<participants>";
  {
    const std::vector<boost::shared_ptr<Participant> >& participants
      = parameters.GetParticipants();
    BOOST_FOREACH(const boost::shared_ptr<Participant>& p,participants)
    {
      os << (*p);
    }
  }
  os
    << "</participants>"
    << (*parameters.GetAssignPayoff())
    << (*parameters.GetChat())
    << (*parameters.GetChooseAction())
    << (*parameters.GetGroupDynamics())
    << (*parameters.GetFinished())
    << (*parameters.GetViewResultsAll())
    << (*parameters.GetViewResultsGroup())
    << (*parameters.GetViewResultsVoting())
    << (*parameters.GetVoting())
    << "</parameters>";

  return os;
}
//---------------------------------------------------------------------------
