//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#ifndef PARAMETERS_H
#define PARAMETERS_H
//---------------------------------------------------------------------------
#include <string>
#include <vector>
//---------------------------------------------------------------------------
#include <boost/scoped_ptr.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/noncopyable.hpp>
//---------------------------------------------------------------------------
#include "forward_declarations.h"
//---------------------------------------------------------------------------
///\brief
///Parameters for an experiment
struct Parameters : public boost::noncopyable
{
  ///Create a default-constructed Parameters
  Parameters();

  ///Create a default non-logged in Participant
  static boost::shared_ptr<Participant> CreateDefaultParticipant();

  ///Deletes all Participant instances
  void DeleteParticipants();

  ///Set a Parameter from file
  void ReadFromFile(const std::string& filename);

  ///Get the ParametersAssignPayoff as a read-and-write pointer
  ParametersAssignPayoff * GetAssignPayoff();
  ///Get the ParametersAssignPayoff as a read-only pointer
  const ParametersAssignPayoff * GetAssignPayoff() const;
  ///Get the ParametersChat as a read-and-write pointer
  ParametersChat * GetChat();
  ///Get the ParametersChat as a read-only pointer
  const ParametersChat * GetChat() const;
  ///Get the ParametersChooseAction as a read-and-write pointer
  ParametersChooseAction * GetChooseAction();
  ///Get the ParametersChooseAction as a read-only pointer
  const ParametersChooseAction * GetChooseAction() const;
  ///Get the ParametersFinished as a read-and-write pointer
  ParametersFinished * GetFinished();
  ///Get the ParametersFinished as a read-only pointer
  const ParametersFinished * GetFinished() const;
  ///Get the ParametersGroupDynamics as a read-and-write pointer
  ParametersGroupDynamics * GetGroupDynamics();
  ///Get the ParametersGroupDynamics as a read-only pointer
  const ParametersGroupDynamics * GetGroupDynamics() const;
  ///Get the ParametersViewResultsAll as a read-and-write pointer
  ParametersViewResultsAll * GetViewResultsAll();
  ///Get the ParametersViewResultsAll as a read-only pointer
  const ParametersViewResultsAll * GetViewResultsAll() const;
  ///Get the ParametersViewResultsGroup as a read-and-write pointer
  ParametersViewResultsGroup * GetViewResultsGroup();
  ///Get the ParametersViewResultsGroup as a read-only pointer
  const ParametersViewResultsGroup * GetViewResultsGroup() const;
  ///Get the ParametersViewResultsVoting as a read-and-write pointer
  ParametersViewResultsVoting * GetViewResultsVoting();
  ///Get the ParametersViewResultsVoting as a read-only pointer
  const ParametersViewResultsVoting * GetViewResultsVoting() const;
  ///Get the ParametersVoting as a read-and-write pointer
  ParametersVoting * GetVoting();
  ///Get the ParametersVoting as a read-only pointer
  const ParametersVoting * GetVoting() const;

  ///Get the Participants as a read-only reference
  const std::vector<boost::shared_ptr<Participant> >& GetParticipants() const { return m_participants; }
  ///Get the Participants as a reference
  std::vector<boost::shared_ptr<Participant> >& GetParticipants() { return m_participants; }

  ///Add Participants to Parameters
  void AddParticipant(boost::shared_ptr<Participant> participant);

  private:
  ///Only allow a Boost smart pointer to delete Administrator
  //to prevent the following trouble,
  //cited from http://www.boost.org/libs/utility/checked_delete.html:
  //The C++ Standard allows, in 5.3.5/5, pointers to incomplete
  //class types to be deleted with a delete-expression.
  //When the class has a non-trivial destructor, or a class-specific operator
  //delete, the behavior is undefined. Some compilers issue a warning when an
  //incomplete type is deleted, but unfortunately, not all do, and programmers
  //sometimes ignore or disable warnings.
  ~Parameters() {}
  ///Only let smart pointers delete Parameters
  //Template syntax from Herb Sutter. Exceptional C++ style. 2005. ISBN: 0-201-76042-8. Item 8: 'Befriending templates'.
  friend void boost::checked_delete<>(Parameters*);


  ///Unassigned Participant colection
  std::vector<boost::shared_ptr<Participant> > m_participants;

  ///The ParametersAssignPayoff
  boost::scoped_ptr<ParametersAssignPayoff> m_assign_payoff;
  ///The ParametersChat
  boost::scoped_ptr<ParametersChat> m_chat;
  ///The ParametersChooseAction
  boost::scoped_ptr<ParametersChooseAction> m_choose_action;
  ///The ParametersFinished
  boost::scoped_ptr<ParametersFinished> m_finished;
  ///The ParametersGroupDynamics
  boost::scoped_ptr<ParametersGroupDynamics> m_group_dynamics;
  ///The ParametersViewResultsAll
  boost::scoped_ptr<ParametersViewResultsAll> m_view_results_all;
  ///The ParametersViewResultsGroup
  boost::scoped_ptr<ParametersViewResultsGroup> m_view_results_group;
  ///The ParametersViewResultsVoting
  boost::scoped_ptr<ParametersViewResultsVoting> m_view_results_voting;
  ///The ParametersVoting
  boost::scoped_ptr<ParametersVoting> m_voting;


  ///Parse a line in a Parameter file
  void Parse(const std::string& s);

  //Helper functions
  ///Check if a file exists
  static bool FileExists(const std::string& filename);
  ///Convert a file to a std::vector
  static const std::vector<std::string> FileToVector(const std::string& filename);
  ///Split a std::string
  static const std::vector<std::string> SeperateString(
    const std::string& input,const char seperator);
};
//---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& os,const Parameters& parameters);
//---------------------------------------------------------------------------
#endif // PARAMETERS_H
