//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#ifndef PARAMETERSGROUPDYNAMICS_H
#define PARAMETERSGROUPDYNAMICS_H
//---------------------------------------------------------------------------
#include <iosfwd>
#include <string>
//---------------------------------------------------------------------------
#include <boost/checked_delete.hpp>
#include <boost/noncopyable.hpp>
//---------------------------------------------------------------------------
#include "forward_declarations.h"
//---------------------------------------------------------------------------
///\brief
///Parameters for the group dynamics state
struct ParametersGroupDynamics
{
  ParametersGroupDynamics();

  ///Types of group dynamics
  enum Type { static_dynamics, dynamic };

  ///Get the type of group dynamics
  Type GetType() const { return m_type; }

  ///Get the message shown to an assigned participant
  const std::string& GetMessageAssigned() const { return m_message_assigned; }

  ///Get the message shown to an unassigned participant
  const std::string& GetMessageUnassigned() const { return m_message_unassigned; }

  ///Set the message shown to an assigned participant
  void SetMessageAssigned(const std::string& message);

  ///Set the message shown to an unassigned participant
  void SetMessageUnassigned(const std::string& message);

  ///Set the type of group dynamics
  void SetType(const Type type) { m_type = type; }

  ///Parse a line
  void Parse(const std::string& s);

  private:
  ///Only allow a Boost smart pointer to delete ParametersGroupDynamics
  //to prevent the following trouble,
  //cited from http://www.boost.org/libs/utility/checked_delete.html:
  //The C++ Standard allows, in 5.3.5/5, pointers to incomplete
  //class types to be deleted with a delete-expression.
  //When the class has a non-trivial destructor, or a class-specific operator
  //delete, the behavior is undefined. Some compilers issue a warning when an
  //incomplete type is deleted, but unfortunately, not all do, and programmers
  //sometimes ignore or disable warnings.
  ~ParametersGroupDynamics() {}
  ///Only let smart pointers delete ParametersGroupDynamics
  //Template syntax from Herb Sutter. Exceptional C++ style. 2005. ISBN: 0-201-76042-8. Item 8: 'Befriending templates'.
  friend void boost::checked_delete<>(ParametersGroupDynamics*);

  //group_dynamics_type is the type of group dynamics
  //- static : groups do not grow or die
  //- dynamic : best groups grow, worst group dies when best group splits up
  Type m_type;

  ///The message at the group dynamics screen when assigned to a group
  std::string m_message_assigned;

  ///The message at the group dynamics screen when not yet assigned to a group
  std::string m_message_unassigned;

  ///Conver a std::string to lower case
  static const std::string StrToLower(std::string s);

};
//---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& os,const ParametersGroupDynamics& parameters);
//---------------------------------------------------------------------------
#endif // PARAMETERSGROUPDYNAMICS_H
