//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <algorithm>
#include <cassert>
#include <iostream>
#include <stdexcept>
//---------------------------------------------------------------------------
#include <boost/lambda/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/numeric/conversion/cast.hpp>
//---------------------------------------------------------------------------
#include "parametersvoting.h"
#include "votingoption.h"
//---------------------------------------------------------------------------
ParametersVoting::ParametersVoting()
  : m_duration(10),
    //m_options(CreateDefaultOptions()),
    m_wait(true)
{

}
//---------------------------------------------------------------------------
const std::vector<boost::shared_ptr<VotingOption> >
  ParametersVoting::CreateDefaultOptions()
{
  std::vector<boost::shared_ptr<VotingOption> > v;
  {
    boost::shared_ptr<VotingOption> p(new VotingOption(0.0,0.0,"No supervisor"));
    v.push_back(p);
  }
  {
    boost::shared_ptr<VotingOption> p(new VotingOption(0.5,1.0,"Lenient supervisor"));
    v.push_back(p);
  }
  {
    boost::shared_ptr<VotingOption> p(new VotingOption(1.0,2.0,"Rigorous supervisor"));
    v.push_back(p);
  }
  return v;
}
//---------------------------------------------------------------------------
int ParametersVoting::GetDuration() const
{
  assert(m_duration >= 0);
  return m_duration;
}
//---------------------------------------------------------------------------
///GetNumberOfOptions returns the number of options to vote for
/*
int ParametersVoting::GetNumberOfOptions() const
{
  const int n
    = boost::numeric_cast<int>(m_options.size());
  assert(n > 1
    && "There should be at least be two voting options");
  return n;
}
*/
//---------------------------------------------------------------------------
///Get the descriptions of the options to vote for only
const std::vector<std::string> ParametersVoting::GetVoteDescriptions() const
{
  std::vector<std::string> w;
  std::transform(
    m_options.begin(),
    m_options.end(),
    std::back_inserter(w),
    boost::lambda::bind(&VotingOption::GetDescription, *boost::lambda::_1)
    );
  return w;
}
//---------------------------------------------------------------------------
///Parse a line
void ParametersVoting::Parse(const std::string& s)
{
  if (s.size() > 9 && s.substr(0,9) == "duration=")
  {
    const std::string t = s.substr(9,s.size()-9);
    try
    {
      boost::lexical_cast<int>(t);
    }
    catch (boost::bad_lexical_cast&)
    {
      throw std::runtime_error("voting_duration must be an integer");
    }
    const int time = boost::lexical_cast<int>(t);
    if (time < 0) throw std::runtime_error("voting_duration must be zero or posive");
    SetDuration(time);
    return;
  }


  if (s.size() > 5 && s.substr(0,5) == "wait=")
  {
    const std::string t = s.substr(5,s.size()-5);
    if (t=="y" || t=="Y" || t=="1" || t == "n" || t == "N" || t =="0")
    {
      if (t=="y" || t=="Y" || t=="1")
      {
        SetWait(true);
      }
      else
      {
        SetWait(false);
      }
    }
    else
    {
      throw std::runtime_error("vote_wait me be either \'y\' or \'n\'");
    }
    return;
  }


  //if (s.size() > 13 && s.substr(0,13) == "descriptions=")
  //{
  //  const std::string t = s.substr(13,s.size() - 13);
  //  const std::vector<std::string> v = SeperateString(t,',');
  //  this->SetVoteDescriptions(v);
  //}


  //Participants
  if (s.size() > 7 && s.substr(0,7) == "option=")
  {
    const std::string t = s.substr(7,s.size() - 7);
    const std::vector<std::string> v = SeperateString(t,',');
    if (v.size() != 3)
    {
      std::clog
        << "Incorrectly formed vote_option line: " << s << '\n'
        << "Must consist of [description][chance][cost]\n"
        << "For example \'Lenient supervisor,0.5,1.0\'\n";
      return;
    }
    assert(v.size() == 3 && "vote_options must have two elements");

    const std::string description = v[0];
    const std::string chance_str = v[1];
    const std::string cost_str = v[2];

    try
    {
      boost::lexical_cast<double>(chance_str);
    }
    catch (std::logic_error&)
    {
      std::clog
        << "Incorrectly formed chance: "
        << chance_str
        << '\n';
      return;
    }
    try
    {
      boost::lexical_cast<double>(cost_str);
    }
    catch (std::logic_error&)
    {
      std::clog
        << "Incorrectly formed cost: "
        << cost_str
        << '\n';
      return;
    }

    const double chance = boost::lexical_cast<double>(chance_str);
    const double cost = boost::lexical_cast<double>(cost_str);
    boost::shared_ptr<VotingOption> p(
      new VotingOption(chance,cost,description));
    m_options.push_back(p);
    //std::clog << "Added an option!\n";
    return;
  }

}
//---------------------------------------------------------------------------
///SeperateString splits a std::string
//From http://www.richelbilderbeek.nl/CppSeperateString.htm
const std::vector<std::string> ParametersVoting::SeperateString(
  const std::string& input,
  const char seperator)
{
  std::istringstream is(input);
  std::vector<std::string> v;
  for (
    std::string sub;
    std::getline(is, sub, seperator);
    v.push_back(sub))
  {
    //Empty for loop
  }
  return v;
}
//---------------------------------------------------------------------------
void ParametersVoting::SetDuration(const int time)
{
  m_duration = time;
  assert(m_duration >= 0);
}
//---------------------------------------------------------------------------
void ParametersVoting::SetOptions(
  const std::vector<boost::shared_ptr<VotingOption> >& options)
{
  assert(options.size() >= 2
    && "A participant must vote between at least two options");
  m_options = options;
}
//---------------------------------------------------------------------------
///Set if there is waited for all before going on
void ParametersVoting::SetWait(const bool wait)
{
  m_wait = wait;
}
//---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& os,const ParametersVoting& parameters)
{
  os
    << "<parametersvoting>"
    << "<time>"
    << parameters.GetDuration()
    << "</time>";
  const int n_actions = boost::numeric_cast<int>(
    parameters.GetOptions().size());
  for (int i=0; i!=n_actions;++i)
  {
    os
      << "<option" << i << ">"
      << (*(parameters.GetOptions()[i]))
      << "</option" << i << ">";
  }
  os
    << "</parametersvoting>";

  return os;
}
//---------------------------------------------------------------------------


