//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <cassert>
#include <iostream>
#include <numeric>
//---------------------------------------------------------------------------
#include <boost/regex.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/numeric/conversion/cast.hpp>
//---------------------------------------------------------------------------
#include "groupassigner.h"
#include "ipaddress.h"
#include "participant.h"
#include "participantstate.h"
#include "participantstateassignpayoff.h"
#include "participantstatechat.h"
#include "participantstatechooseaction.h"
#include "participantstatefinished.h"
#include "participantstategroupdynamics.h"
#include "participantstateloggedin.h"
#include "participantstatenotloggedin.h"
#include "participantstateviewresultsall.h"
#include "participantstateviewresultsgroup.h"
#include "participantstateviewresultsvoting.h"
#include "participantstatevoting.h"
#include "state.h"
//---------------------------------------------------------------------------
Participant::Participant(
  const char chat_tag,
  boost::shared_ptr<GroupAssigner> group_assigner)
  : m_chat_tag(chat_tag),
    m_group_assigner(group_assigner),
    m_id(0),
    m_state(new ParticipantStateNotLoggedIn(this)),
    m_state_assign_payoff(new ParticipantStateAssignPayoff(this)),
    m_state_chat(new ParticipantStateChat(this)),
    m_state_choose_action(new ParticipantStateChooseAction(this)),
    m_state_finished(new ParticipantStateFinished(this)),
    m_state_group_dynamics(new ParticipantStateGroupDynamics(this)),
    m_state_logged_in(new ParticipantStateLoggedIn(this)),
    m_state_not_logged_in(new ParticipantStateNotLoggedIn(this)),
    m_state_view_results_all(new ParticipantStateViewResultsAll(this)),
    m_state_view_resuls_group(new ParticipantStateViewResultsGroup(this)),
    m_state_view_results_voting(new ParticipantStateViewResultsVoting(this)),
    m_state_voting(new ParticipantStateVoting(this))
{
  #ifdef LOG_PARTICIPANT_LOGIN
  std::clog  << (*this) << "\n";
  #endif
  assert(m_id == 0 && "Assume ID is unassigned");
  assert(m_group_assigner);
  assert(m_state);

  assert(!CanGetId());
}
//---------------------------------------------------------------------------
Participant::~Participant()
{
  //std::clog << "Participant::~Participant() for #" << GetId() << '\n' ;
}
//---------------------------------------------------------------------------
void Participant::AppendChat(const std::string& s)
{
  assert(!s.empty() && "Assume not empty chat messages");
  assert(s[s.size()-1]!='\n' && "Assume no newlines");
  assert(!m_chat.empty());

  m_chat.back().push_back(s);
}
//---------------------------------------------------------------------------
///Assign the Participant an ID
void Participant::AssignId(const int id)
{
  assert(id > 0 && "ID\'s must be positive non-zero values");
  m_id = id;
}
//---------------------------------------------------------------------------
///Assign the payoff the Participant receives from the chosen action
void Participant::AssignPayoff(const double payoff)
{
  m_payoffs.push_back(payoff);
}
//---------------------------------------------------------------------------
///Check if the Participant is assigned an ID already
///if m_id equals zero, the Participant is not assigned an ID yet
bool Participant::CanGetId() const
{
  assert(m_id >= 0);
  return m_id != 0;
}
//---------------------------------------------------------------------------
bool Participant::CanGetIpAddress() const
{
  return m_ip_address;
}
//---------------------------------------------------------------------------
///Lets the Participant choose an action
void Participant::ChooseAction(const ChooseActionOption * const option)
{
  assert(option);
  //assert(!HasChosenAction()
  //  && "A Participant can only choose an action once per round");

  m_actions.push_back(option);

  //assert(HasChosenAction()
  //  && "A Participant should chosen an action now");
}
//---------------------------------------------------------------------------
///Get the Participant his/her unique ID
int Participant::GetId() const
{
  assert(m_id > 0 && "ID\'s must be positive non-zero values");
  return m_id;
}
//---------------------------------------------------------------------------
///Get the Participant's IP address
const SafeIpAddress * Participant::GetIpAddress() const
{
  //assert(m_ip_address);
  return m_ip_address.get();
}
//---------------------------------------------------------------------------
///Get the Participant his/her total payoff
double Participant::GetPayoffTotal() const
{
  return std::accumulate(m_payoffs.begin(),m_payoffs.end(),0.0);
}
//---------------------------------------------------------------------------
///Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
ParticipantState * Participant::GetState()
{
  return const_cast<ParticipantState*>(
    const_cast<const Participant&>(*this).GetState());
}
//---------------------------------------------------------------------------
const ParticipantState * Participant::GetState() const
{
  assert(m_state);
  return m_state;
}
//---------------------------------------------------------------------------
void Participant::SetIpAddress(const SafeIpAddress * const ip_address)
{
  assert(!CanGetIpAddress() && "Cannot change the IP address of a participant if it is valid already");

  m_ip_address.reset(new SafeIpAddress(ip_address->Get()));
}
//---------------------------------------------------------------------------
///SetState sets the ParticipantState of the Participant
void Participant::SetState(const State * const state)
{
  if (dynamic_cast<const StateAssignPayoff*>(state))
  {
    m_state = m_state_assign_payoff.get();
  }
  else if (dynamic_cast<const StateChat*>(state))
  {
    m_state = m_state_chat.get();
  }
  else if (dynamic_cast<const StateChooseAction*>(state))
  {
    m_state = m_state_choose_action.get();
  }
  else if (dynamic_cast<const StateFinished*>(state))
  {
    m_state = m_state_finished.get();
  }
  else if (dynamic_cast<const StateGroupDynamics*>(state))
  {
    m_state = m_state_group_dynamics.get();
  }
  else if (dynamic_cast<const StateLoggedIn*>(state))
  {
    m_state = m_state_logged_in.get();
  }
  else if (dynamic_cast<const StateNotLoggedIn*>(state))
  {
    m_state = m_state_not_logged_in.get();
  }
  else if (dynamic_cast<const StateViewResultsAll*>(state))
  {
    m_state = m_state_view_results_all.get();
  }
  else if (dynamic_cast<const StateViewResultsGroup*>(state))
  {
    m_state = m_state_view_resuls_group.get();
  }
  else if (dynamic_cast<const StateViewResultsVoting*>(state))
  {
    m_state = m_state_view_results_voting.get();
  }
  else if (dynamic_cast<const StateVoting*>(state))
  {
    m_state = m_state_voting.get();
  }
  else
  {
    assert(!"Should not get here: unimplemented state for Participant");
  }
}
//---------------------------------------------------------------------------
///StartAssignPayoff lets all Participants know
///that their payoffs are assigned.
//void Participant::StartAssignPayoff()
//{
//  ++m_n_payoffs_assigned;
//}
//---------------------------------------------------------------------------
///StartChat lets the Participant know that he/she starts a new chat phase
void Participant::StartChat()
{
  //assert(!CanGetLastChat() || HasChatted());

  //++m_n_chats;
  m_chat.push_back(std::vector<std::string>());
  //assert(!HasChatted());
}
//---------------------------------------------------------------------------
///StartChooseAction lets all Participants know that they have to
///choose an (IPGG) action
//void Participant::StartChooseAction()
//{
//  ++m_n_actions_chosen;
//}
//---------------------------------------------------------------------------
///StartVoting lets all Participants know that they have to vote
//void Participant::StartVoting()
//{
//  ++m_n_votes;
//}
//---------------------------------------------------------------------------
///ToStr creates a one-line string viewed by an Administrator
const std::string Participant::ToAdminStr() const
{
  return this->GetState()->ToAdminStr();
}
//---------------------------------------------------------------------------
///Lets the Participant vote
void Participant::Vote(const int vote)
{
  //assert(!HasVoted() && "A Participant can only vote once per round");

  m_votes.push_back(vote);

  //assert(HasVoted() && "A Participant should have voted now");
}
//---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& os,const Participant& p)
{
  os
    << "<participant>"
    << "<id>"
    << (p.CanGetId()
      ? boost::lexical_cast<std::string>(p.GetId())
      : std::string("N/A"))
    << "</id>"
    << "<ip_address>"
    << (p.CanGetIpAddress()
      ? p.GetIpAddress()->Get()
      : std::string("*"))
    << "</ip_address>"
    //<< "<group>"
    //<< (p.CanGetGroup()
    //  ? boost::lexical_cast<std::string>(p.GetGroup())
    //  : std::string("N/A"))
    //<< std::string("</group>")
    //<< std::string("<is_participating>")
    //<< p.IsParticipating()
    //<< std::string("</is_participating>")
    << "<state>"
    << (p.GetState()->ToStr())
    << "</state>"
    << "</participant>";
  return os;
  }
//---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& os,const boost::shared_ptr<Participant>& p)
{
  assert(p);
  os << (*(p.get()));
  return os;
}
//---------------------------------------------------------------------------
bool operator<(const Participant& lhs,const Participant& rhs)
{
  assert(lhs.GetId() != rhs.GetId() && "Assume each participant has a unique ID");
  return lhs.GetId() < rhs.GetId();
}
//---------------------------------------------------------------------------
