//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#ifndef PARTICIPANTDIALOG_H
#define PARTICIPANTDIALOG_H
//---------------------------------------------------------------------------
#include <string>
//---------------------------------------------------------------------------
#include <boost/noncopyable.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/shared_ptr.hpp>
//---------------------------------------------------------------------------
#include <Wt/WContainerWidget>
//---------------------------------------------------------------------------
#include "forward_declarations.h"
//---------------------------------------------------------------------------
///\brief
///ParticipantDialog handles the user interface for a Participant.
///
///ParticipantDialog is an Observer of Participant and does
///not alter Participant itself. It does so by connecting to
///the Server.
struct ParticipantDialog : public Wt::WContainerWidget// , boost::noncopyable
{
  ///Construct a ParticipantDialog and use its IP address
  ParticipantDialog(const SafeIpAddress * const ip_address);

  ///Check if a valid read-only pointer to the Participant can be get
  bool CanGetParticipant() const;

  ///Do something random with the UI, used by DebugDialog
  void DoSomethingRandom();

  ///Get this ParticipantDialog its IP address
  const SafeIpAddress * GetIpAddress() const;

  ///Obtain a read-only pointer to the Participant
  const Participant * GetParticipant() const;

  ///Assign a Participant to this ParticipantDialog
  void SetParticipant(const Participant * const participant);

  ///Get the ParticipantDialog its current ParticipantDialogState
  ParticipantDialogState * GetCurrentState() const;
  ///Get the ParticipantDialog its ParticipantDialogStateAssignPayoff
  ParticipantDialogStateAssignPayoff * GetStateAssignPayoff() const;
  ///Get the ParticipantDialog its ParticipantDialogStateChat
  ParticipantDialogStateChat * GetStateChat() const;
  ///Get the ParticipantDialog its ParticipantDialogStateChooseAction
  ParticipantDialogStateChooseAction * GetStateChooseAction() const;
  ///Get the ParticipantDialog its ParticipantDialogStateFinished
  ParticipantDialogStateFinished * GetStateFinished() const;
  ///Get the ParticipantDialog its ParticipantDialogStateGroupDynamics
  ParticipantDialogStateGroupDynamics * GetStateGroupDynamics() const;
  ///Get the ParticipantDialog its ParticipantDialogStateLoggedIn
  ParticipantDialogStateLoggedIn * GetStateLoggedIn() const;
  ///Get the ParticipantDialog its ParticipantDialogStateViewResultsAll
  ParticipantDialogStateNotLoggedIn * GetStateNotLoggedIn() const;
  ///Get the ParticipantDialog its ParticipantDialogStateViewResultsAll
  ParticipantDialogStateViewResultsAll * GetStateViewResultsAll() const;
  ///Get the ParticipantDialog its ParticipantDialogStateViewResultsGroup
  ParticipantDialogStateViewResultsGroup * GetStateViewResultsGroup() const;
  ///Get the ParticipantDialog its ParticipantDialogStateViewResultsVoting
  ParticipantDialogStateViewResultsVoting * GetStateViewResultsVoting() const;
  ///Get the ParticipantDialog its ParticipantDialogStateVoting
  ParticipantDialogStateVoting * GetStateVoting() const;

  ///Set the ParticipantDialogState of this ParticipantDialog
  void SetState(ParticipantDialogState * const state);

  private:
  ParticipantDialog( const ParticipantDialog& );
  const ParticipantDialog& operator=( const ParticipantDialog& );

  ///\brief Only allow a Boost smart pointer to delete ParticipantDialog
  ///
  ///This prevents the following trouble,
  ///cited from http://www.boost.org/libs/utility/checked_delete.html:
  ///The C++ Standard allows, in 5.3.5/5, pointers to incomplete
  ///class types to be deleted with a delete-expression.
  ///When the class has a non-trivial destructor, or a class-specific operator
  ///delete, the behavior is undefined. Some compilers issue a warning when an
  ///incomplete type is deleted, but unfortunately, not all do, and programmers
  ///sometimes ignore or disable warnings.
  ~ParticipantDialog();
  ///Only allow a Boost smart pointer to delete ParticipantDialog
  //Template syntax from Herb Sutter. Exceptional C++ style. 2005. ISBN: 0-201-76042-8. Item 8: 'Befriending templates'.
  friend void boost::checked_delete<>(ParticipantDialog*);

  ///The IP address this ParticipantDialog is recognized with
  boost::scoped_ptr<SafeIpAddress> m_ip_address;

  ///Read-only Participant pointer, obtained from server
  const Participant * m_participant;

  ///m_state is the current State of the dialog

  ///The ParticipantDialog its current ParticipantDialogState
  ParticipantDialogState * m_state;
  ///The ParticipantDialog its ParticipantDialogStateAssignPayoff
  boost::scoped_ptr<ParticipantDialogStateAssignPayoff> m_state_assign_payoff;
  ///The ParticipantDialog its ParticipantDialogStateChat
  boost::scoped_ptr<ParticipantDialogStateChat> m_state_chat;
  ///The ParticipantDialog its ParticipantDialogStateChooseAction
  boost::scoped_ptr<ParticipantDialogStateChooseAction> m_state_choose_action;
  ///The ParticipantDialog its ParticipantDialogStateFinished
  boost::scoped_ptr<ParticipantDialogStateFinished> m_state_finished;
  ///The ParticipantDialog its ParticipantDialogStateGroupDynamics
  boost::scoped_ptr<ParticipantDialogStateGroupDynamics> m_state_group_dynamics;
  ///The ParticipantDialog its ParticipantDialogStateLoggedIn
  boost::scoped_ptr<ParticipantDialogStateLoggedIn> m_state_logged_in;
  ///The ParticipantDialog its ParticipantDialogStateNotLoggedIn
  boost::scoped_ptr<ParticipantDialogStateNotLoggedIn> m_state_not_logged_in;
  ///The ParticipantDialog its ParticipantDialogStateViewResultsAll
  boost::scoped_ptr<ParticipantDialogStateViewResultsAll> m_state_view_results_all;
  ///The ParticipantDialog its ParticipantDialogStateViewResultsGroup
  boost::scoped_ptr<ParticipantDialogStateViewResultsGroup> m_state_view_results_group;
  ///The ParticipantDialog its ParticipantDialogStateViewResultsVoting
  boost::scoped_ptr<ParticipantDialogStateViewResultsVoting> m_state_view_results_voting;
  ///The ParticipantDialog its ParticipantDialogStateVoting
  boost::scoped_ptr<ParticipantDialogStateVoting> m_state_voting;

  ///The ParticipantDialog its timer
  const boost::scoped_ptr<Wt::WTimer> m_timer;

  ///Respond to the Server its timer
  //void OnServerTimer();

  ///Respond to ParticipantDialog its own timer
  void OnTimer();
};
//---------------------------------------------------------------------------
#endif // PARTICIPANTDIALOG_H
