//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <cstdio>
#include <ctime>
#include <fstream>
#include <iostream>
#include <string>
//---------------------------------------------------------------------------
#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <boost/lambda/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/numeric/conversion/cast.hpp>
//---------------------------------------------------------------------------
#include "administrator.h"
#include "group.h"
#include "groupfinished.h"
#include "grouploggedin.h"
#include "groupnotloggedin.h"
#include "groupparticipating.h"
#include "groups.h"
#include "state.h"
#include "logfile.h"
#include "parameters.h"
#include "parametersassignpayoff.h"
#include "parameterschat.h"
#include "parameterschooseaction.h"
#include "parametersfinished.h"
#include "parametersgroupdynamics.h"
#include "parametersviewresultsall.h"
#include "parametersviewresultsgroup.h"
#include "parametersviewresultsvoting.h"
#include "parametersvoting.h"
#include "participant.h"
#include "repeatassigner.h"
#include "resources.h"
#include "stopwatch.h"
#include "test.h"
#include "server.h"
#include "serverstate.h"
#include "serverstateassignpayoff.h"
#include "serverstatechat.h"
#include "serverstatechooseaction.h"
#include "serverstatefinished.h"
#include "serverstategroupdynamics.h"
#include "serverstateviewresultsall.h"
#include "serverstateviewresultsgroup.h"
#include "serverstateviewresultsvoting.h"
#include "serverstatevoting.h"
#include "serverstatewaiting.h"
#include "trace.h"
//---------------------------------------------------------------------------
///The Server Singleton its only instance
Server * Server::m_instance = 0;
//---------------------------------------------------------------------------
///m_poll_time is the time the server emits m_signal_timer
const int Server::m_poll_time = 100;
//---------------------------------------------------------------------------
///The filename of the default parameter file
const std::string Server::m_default_parameters_filename = "parameters_default.txt";
//---------------------------------------------------------------------------
Server::Server()
  : m_groups(new Groups),
    m_last_id_administrator(0),
    m_log(new LogFile),
    m_parameters(new Parameters),
    m_state(0),
    m_state_assign_payoff(new ServerStateAssignPayoff(this)),
    m_state_chat(new ServerStateChat(this)),
    m_state_choose_action(new ServerStateChooseAction(this)),
    m_state_finished(new ServerStateFinished(this)),
    m_state_group_dynamics(new ServerStateGroupDynamics(this)),
    m_state_view_results_all(new ServerStateViewResultsAll(this)),
    m_state_view_results_group(new ServerStateViewResultsGroup(this)),
    m_state_view_results_voting(new ServerStateViewResultsVoting(this)),
    m_state_voting(new ServerStateVoting(this)),
    m_state_waiting(new ServerStateWaiting(this))
    //m_unassigned_participants(new GroupNotParticipating)
{
  //Create resources
  {
    boost::scoped_ptr<Resources>(new Resources);
  }

  Reset();
}
//---------------------------------------------------------------------------
///FileExists checks if a certain file exists
///From http://www.richelbilderbeek.nl/CppFileExists.htm
bool Server::FileExists(const std::string& filename)
{
  std::fstream f;
  f.open(filename.c_str(),std::ios::in);
  return f.is_open();
}
//---------------------------------------------------------------------------
///FileToVector reads a file and converts it to a std::vector<std::string>
///From http://www.richelbilderbeek.nl/CppFileToVector.htm
const std::vector<std::string> Server::FileToVector(const std::string& filename)
{
  assert(FileExists(filename));
  std::vector<std::string> v;
  std::ifstream in(filename.c_str());
  std::string s;
  for (int i=0; !in.eof(); ++i)
  {
    std::getline(in,s);
    v.push_back(s);
  }
  return v;
}
//---------------------------------------------------------------------------
ServerState * Server::GetCurrentState() const
{
  assert(m_state);
  return m_state;
}
//---------------------------------------------------------------------------
///Deletes all Participant instances
void Server::DeleteParticipants()
{
  m_groups.reset(new Groups);
  assert(m_parameters);
  m_parameters->DeleteParticipants();
}
//---------------------------------------------------------------------------
Server * Server::Get()
{
  if (!m_instance) m_instance = new Server;
  assert(m_instance);
  return m_instance;
}
//---------------------------------------------------------------------------
///Obtain a read-only pointer to Groups
const Groups * Server::GetGroups() const
{
  assert(m_groups);
  return m_groups.get();
}
//---------------------------------------------------------------------------
///Obtain a read-and-write pointer to Groups
//Avoiding duplication in const and non-const member functions.
//Scott Meyers. Effective C++ (3rd edition). ISBN: 0-321-33487-6. Item 3,
//paragraph 'Avoid duplication in const and non-const member functions'
Groups * Server::GetGroups()
{
  return const_cast<Groups*>(
    const_cast<const Server&>(*this).GetGroups());
}
//---------------------------------------------------------------------------

LogFile * Server::GetLog() const
{
  //#ifdef Server_USE_MUTEX
  //boost::mutex::scoped_lock lock(m_mutex);
  //#endif

  assert(m_log);
  return m_log.get();
}
//---------------------------------------------------------------------------
///GetNewGroupNumber returns a number that is not an existing group number
/*
int Server::GetNewGroupNumber() const
{

  int number = -1;
  //Find the highest number and add one
  BOOST_FOREACH(const boost::shared_ptr<Participant>& p, m_participants)
  {
    if (p->CanGetGroup() && p->GetGroup() > number)
    {
      number = p->GetGroup();
    }
  }
  assert(number != -1);
  return number + 1;
}
*/
//---------------------------------------------------------------------------
const Parameters * Server::GetParameters() const
{
  assert(m_parameters);
  return m_parameters.get();
}
//---------------------------------------------------------------------------
///Obtain the states of the Participant collection in a text form,
///for WtAdministratorDialog
/*
const std::vector<std::string> Server::GetParticipantStates() const
{
  assert(GetCurrentState());

  return GetCurrentState()->GetParticipantStates();
}
*/
//---------------------------------------------------------------------------
ServerStateAssignPayoff * Server::GetStateAssignPayoff() const
{
  assert(m_state_assign_payoff);
  return m_state_assign_payoff.get();
}
//---------------------------------------------------------------------------
ServerStateChat * Server::GetStateChat() const
{
  assert(m_state_chat);
  return m_state_chat.get();
}
//---------------------------------------------------------------------------
ServerStateChooseAction * Server::GetStateChooseAction() const
{
  assert(m_state_choose_action);
  return m_state_choose_action.get();
}
//---------------------------------------------------------------------------
ServerStateFinished * Server::GetStateFinished() const
{
  assert(m_state_finished);
  return m_state_finished.get();
}
//---------------------------------------------------------------------------
ServerStateGroupDynamics * Server::GetStateGroupDynamics() const
{
  assert(m_state_group_dynamics);
  return m_state_group_dynamics.get();
}
//---------------------------------------------------------------------------
ServerStateViewResultsAll * Server::GetStateViewResultsAll() const
{
  assert(m_state_view_results_all);
  return m_state_view_results_all.get();
}
//---------------------------------------------------------------------------
ServerStateViewResultsGroup * Server::GetStateViewResultsGroup() const
{
  assert(m_state_view_results_group);
  return m_state_view_results_group.get();
}
//---------------------------------------------------------------------------
ServerStateViewResultsVoting * Server::GetStateViewResultsVoting() const
{
  assert(m_state_view_results_voting);
  return m_state_view_results_voting.get();
}
//---------------------------------------------------------------------------
ServerStateVoting * Server::GetStateVoting() const
{
  assert(m_state_voting);
  return m_state_voting.get();
}
//---------------------------------------------------------------------------
ServerStateWaiting * Server::GetStateWaiting() const
{
  assert(m_state_waiting);
  return m_state_waiting.get();
}
//---------------------------------------------------------------------------
///GetTimeLeft deterimines how much time is spent in this phase
int Server::GetTimeLeft() const
{
  assert(m_state);
  return m_state->GetTimeLeft();
}
//---------------------------------------------------------------------------
///KillGroup removes all participants from a group
///(that one wwith the least success)
/*
void Server::KillGroup(const int index_least_successful)
{
  const int n_groups
    = boost::numeric_cast<int>(m_groups.size());

  for (int i=0; i!=n_groups; ++i)
  {
    if (m_groups[i]->GetId() == index_least_successful)
    {
      std::swap(m_groups[i],m_groups.back());
      m_groups.pop_back();
      return;
    }
  }
}
*/
//---------------------------------------------------------------------------
///Login creates a unique Administrator (with unique ID)
//for each client. This is an unlogged event.
boost::shared_ptr<Administrator> Server::LoginAdministrator()
{
  #ifdef Server_USE_MUTEX
  boost::mutex::scoped_lock lock(m_mutex);
  #endif

  #ifndef NDEBUG
  const int n_administrators_old = m_administrators.size();
  BOOST_FOREACH(const boost::shared_ptr<Administrator>& p,m_administrators)
  {
    assert(p && "Assume all Administrators are valid");
  }
  #endif

  ++m_last_id_administrator;
  const int new_id = m_last_id_administrator;
  assert(new_id > 0);

  #ifndef NDEBUG
  BOOST_FOREACH(const boost::shared_ptr<Administrator>& p,m_administrators)
  {
    assert(p->GetId() != new_id && "Assume new ID is really a unique ID");
  }
  #endif

  boost::shared_ptr<Administrator> new_administrator(
    new Administrator(new_id));
  assert(new_administrator);

  //Keep one copy of the administrator
  m_administrators.insert(new_administrator);

  #ifndef NDEBUG
  const int n_administrators_new = m_administrators.size();
  assert(n_administrators_new == n_administrators_old + 1
    && "Assume the new administrator has really been added to m_administrators");
  #endif

  return new_administrator;
}
//---------------------------------------------------------------------------
///Login creates a unique Participant (with unique ID)
//for each client. This is a logged event.
const Participant * Server::LetLogin(const SafeIpAddress * const ip_address)
{
  boost::mutex::scoped_lock lock(m_mutex);

  //Check for participants that reloaded the page by pressing F5
  if (GetGroups()->CanGetParticipantWithIpAddress(ip_address))
  {
    return GetGroups()->GetParticipantWithIpAddress(ip_address);
  }

  assert(GetGroups()->CanLetLogin(ip_address));

  const Participant * const new_participant
    = m_groups->LetLogin(ip_address);
  assert(new_participant);

  //Logging this event
  m_log->Login(new_participant);

  //New participants try to log in
  return new_participant;
}
//---------------------------------------------------------------------------
///Resets the server to its initial state
void Server::Reset()
{

  m_administrators.clear();
  assert(m_administrators.empty());

  m_last_id_administrator = 0;
  m_log.reset(new LogFile);
  m_parameters.reset(new Parameters);
  m_groups.reset(new Groups);
  GroupParticipating::Reset();

  m_state = 0;
  m_state_assign_payoff.reset(new ServerStateAssignPayoff(this));
  m_state_chat.reset(new ServerStateChat(this));
  m_state_choose_action.reset(new ServerStateChooseAction(this));
  m_state_finished.reset(new ServerStateFinished(this));
  m_state_group_dynamics.reset(new ServerStateGroupDynamics(this));
  m_state_view_results_all.reset(new ServerStateViewResultsAll(this));
  m_state_view_results_group.reset(new ServerStateViewResultsGroup(this));
  m_state_view_results_voting.reset(new ServerStateViewResultsVoting(this));
  m_state_voting.reset(new ServerStateVoting(this));
  m_state_waiting.reset(new ServerStateWaiting(this));

  m_state = m_state_waiting.get();

  assert(m_log);
  assert(m_state);
  assert(m_state_assign_payoff);
  assert(m_state_chat);
  assert(m_state_choose_action);
  assert(m_state_finished);
  assert(m_state_group_dynamics);
  assert(m_state_view_results_all);
  assert(m_state_view_results_group);
  assert(m_state_view_results_voting);
  assert(m_state_voting);
  assert(m_state_waiting);

  assert(GetCurrentState());
  assert(GetStateAssignPayoff());
  assert(GetStateChat());
  assert(GetStateChooseAction());
  assert(GetStateFinished());
  assert(GetStateGroupDynamics());
  assert(GetStateViewResultsAll());
  assert(GetStateViewResultsGroup());
  assert(GetStateViewResultsVoting());
  assert(GetStateVoting());
  assert(GetStateWaiting());

  ///Set the default parameters
  this->SetParameters(m_parameters);

  ///Set the server to the waiting state
  SetState(GetStateWaiting());

  assert(m_state);
  //assert(m_timer);
  assert(m_poll_time > 0);
  //m_timer->setInterval(m_poll_time);
  //m_timer->timeout().connect(this,&Server::OnTimer);
  //m_timer->start();
}
//---------------------------------------------------------------------------
//Nobody logs out: he/she refreshes or has a browser crash
//When a Participant logs out, this event _might_ be called: if
//a client presses refesh, Server::NotifyLogout is _not_ called!
void Server::NotifyLogout(const Participant * const participant)
{
  #ifdef Server_USE_MUTEX
  boost::mutex::scoped_lock lock(m_mutex);
  #endif

  assert(participant);

  //Logging this eventServerState
  m_log->Logout(participant);

  //Do not erase the Participant: it is used again when he/she
  //refreshed his/her browser, so he/she can log in easily
}
//---------------------------------------------------------------------------
void Server::OnTimer()
{
  //#ifdef Server_USE_MUTEX
  //boost::mutex::scoped_lock lock(m_mutex);
  //#endif

  //m_signal_timer.emit();
  assert(m_state);
  TRACE(m_state->ToStr());

  m_state->OnTimer();
}
//---------------------------------------------------------------------------
void Server::SetParameters(boost::shared_ptr<Parameters> parameters)
{
  GroupParticipating::Reset();

  assert(parameters);
  m_parameters = parameters;

  assert(m_log);
  m_log->LogParameters(m_parameters);

  assert(m_state_assign_payoff);
  assert(m_state_chat);
  assert(m_state_choose_action);
  assert(m_state_finished);
  assert(m_state_group_dynamics);
  assert(m_state_view_results_all);
  assert(m_state_view_results_group);
  assert(m_state_view_results_group);
  assert(m_state_view_results_voting);
  assert(m_state_voting);
  m_state_assign_payoff->SetParameters(m_parameters->GetAssignPayoff());
  m_state_chat->SetParameters(m_parameters->GetChat());
  m_state_choose_action->SetParameters(m_parameters->GetChooseAction());
  m_state_finished->SetParameters(m_parameters->GetFinished());
  m_state_group_dynamics->SetParameters(m_parameters->GetGroupDynamics());
  m_state_view_results_all->SetParameters(m_parameters->GetViewResultsAll());
  m_state_view_results_group->SetParameters(m_parameters->GetViewResultsGroup());
  m_state_view_results_voting->SetParameters(m_parameters->GetViewResultsVoting());
  m_state_voting->SetParameters(m_parameters->GetVoting());
  ///m_state_waiting->SetParameters(m_parameters->GetWaiting());

  assert(m_groups);
  m_groups->SetParticipants(m_parameters->GetParticipants());

  SetState(GetStateGroupDynamics());
}
//---------------------------------------------------------------------------
///Set the experiment to another phase
void Server::SetState(ServerState* const new_state)
{
  TRACE_FUNC();

  if (new_state == m_state) return;
  #ifdef Server_USE_MUTEX
  boost::mutex::scoped_lock lock(m_mutex);
  #endif

  assert(new_state);
  m_state = new_state;
  assert(GetCurrentState() == new_state);

  TRACE("X1");

  m_state->Start();

  TRACE("X2");

  m_state->ResetTimeLeft();


  TRACE("X3");

  assert(m_state->GetTimeLeft() >= -1 //-1 instead of 0: perhaps a single clock tick is made
    && "All states must have their times reset");

  m_log->LogExperimentStateChanged(GetCurrentState());
}
//---------------------------------------------------------------------------
///SplitGroup removes three from the five participants in the group
///with the most success
/*
void Server::SplitGroup(const int index_most_successful)
{
  const int new_group_number = GetNewGroupNumber();

  for (int i=0; i!=3; ++i)
  {
    BOOST_FOREACH(const boost::shared_ptr<Participant>& p,m_participants)
    {
      if (p->CanGetGroup() && p->GetGroup() == index_most_successful)
      {
        p->ReassignGroup(new_group_number);
        break;
      }
    }
  }
}
*/
//---------------------------------------------------------------------------
void Server::TestMe()
{
  ///Only test server for the first person that logs in
  static bool has_tested = false;
  if (has_tested) return;
  boost::scoped_ptr<Test> test(new Test);
  has_tested = true;
}
//---------------------------------------------------------------------------
///Test if the timer still works
void Server::WakeUp()
{
  this->OnTimer();
}
//---------------------------------------------------------------------------
