//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#ifndef SERVER_H
#define SERVER_H
//---------------------------------------------------------------------------
#include <set>
#include <vector>
//---------------------------------------------------------------------------
#include <boost/noncopyable.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/thread/mutex.hpp>
//---------------------------------------------------------------------------
#include "forward_declarations.h"
//---------------------------------------------------------------------------
///\brief
///The center of communication.
///
///Server is the center of communication between
///participants and the administrators. It does so by creating
///files adminX.txt for each administrator (where X is the administrator's
///index). These file are appended by Server and read and cleared by
///each participant and administrator. The file log.txt is created that
///allows for full session recovery and will only be read.
///Server is a Singleton.
struct Server : public boost::noncopyable
{
  ///Obtain the instance of the Singleton
  static Server * Get();

  ///Deletes all Participant instances
  void DeleteParticipants();

  ///Let an Administrator log in
  boost::shared_ptr<Administrator> LoginAdministrator();

  ///Let a Participant with a certain IP address log in for the first or later time
  const Participant * LetLogin(const SafeIpAddress * const ip_address);

  ///Obtain the Administrator collection
  const std::set<boost::shared_ptr<Administrator> >& GetAdministrators() const { return m_administrators; }

  ///Obtain a read-only pointer to Groups
  const Groups * GetGroups() const;

  ///Obtain a read-and-write pointer to Groups
  Groups * GetGroups();

  ///Obtain the Participant collection
  //const std::set<boost::shared_ptr<Participant> >& GetParticipants() const { return m_participants; }

  ///Obtain the Log File
  LogFile * GetLog() const;

  ///Obtain the Parameters
  const Parameters * GetParameters() const;

  ///Obtain the states of the Participant collection in a text form,
  ///for WtAdministratorDialog
  //const std::vector<std::string> GetParticipantStates() const;

  ///Sets the Parameters
  void SetParameters(boost::shared_ptr<Parameters> parameters);

  //State management
  ///Get the current/active ServerState
  ServerState * GetCurrentState() const;

  ///Get the ServerStateAssignPayoff
  ServerStateAssignPayoff * GetStateAssignPayoff() const;
  ///Get the ServerStateChat
  ServerStateChat * GetStateChat() const;
  ///Get the ServerStateChooseAction
  ServerStateChooseAction * GetStateChooseAction() const;
  ///Get the ServerStateFinished
  ServerStateFinished * GetStateFinished() const;
  ///Get the ServerStateGroupDynamics
  ServerStateGroupDynamics * GetStateGroupDynamics() const;
  ///Get the ServerStateViewResultsAll
  ServerStateViewResultsAll * GetStateViewResultsAll() const;
  ///Get the ServerStateViewResultsGroup
  ServerStateViewResultsGroup * GetStateViewResultsGroup() const;
  ///Get the ServerStateViewResultsVoting
  ServerStateViewResultsVoting * GetStateViewResultsVoting() const;
  ///Get the ServerStateVoting
  ServerStateVoting * GetStateVoting() const;
  ///Get the ServerStateWaiting
  ServerStateWaiting * GetStateWaiting() const;

  ///Set the current ServerState to its new state
  void SetState(ServerState * const new_state);

  ///GetTimeLeft deterimines how much time is spent in this phase
  int GetTimeLeft() const;

  ///KillGroup removes all participants from a group
  ///(that one wwith the least success)
  //void KillGroup(const int index_least_successful);

  //Nobody ever logs out: he/she just refreshes or has a browser crash
  //When the browser is closed, the server is notified. This is not
  //the case when a browser is refreshed (by pressing F5, for example) :-(
  void NotifyLogout(const Participant * const participant);

  ///Resets the server to its initial state
  void Reset();

  ///SplitGroup removes three from the five participants in the group
  ///with the most success
  //void SplitGroup(const int index_most_successful);

  ///The default filename for loading a Parameters
  static const std::string m_default_parameters_filename;

  ///The time in miliseconds that there is polled by Server
  ///and -too bad- ParticipantDialog and WtAdministratorDialog
  static const int m_poll_time;

  ///Test the Server
  static void TestMe();

  ///Test if the timer still works
  void WakeUp();

  private:
  ///Constructor is private, as Server is a Singleton
  Server();

  ///\brief Only allow a Boost smart pointer to delete Server
  ///
  ///This prevents the following trouble,
  ///cited from http://www.boost.org/libs/utility/checked_delete.html:
  ///The C++ Standard allows, in 5.3.5/5, pointers to incomplete
  ///class types to be deleted with a delete-expression.
  ///When the class has a non-trivial destructor, or a class-specific operator
  ///delete, the behavior is undefined. Some compilers issue a warning when an
  ///incomplete type is deleted, but unfortunately, not all do, and programmers
  ///sometimes ignore or disable warnings.
  ///
  ///\note
  ///Because Server is a Singleton, Server will be deleted only at
  ///the end of the program. Yet, this approach is chosen to be sure
  ///the trouble above is circumvented
  ~Server() {}
  ///Only allow a Boost smart pointer to delete Server
  //Template syntax from Herb Sutter. Exceptional C++ style. 2005. ISBN: 0-201-76042-8. Item 8: 'Befriending templates'.
  friend void boost::checked_delete<>(Server*);

  ///The set of active Participant
  std::set<boost::shared_ptr<Administrator> > m_administrators;

  ///The Groups Facade
  boost::scoped_ptr<Groups> m_groups;

  ///The only instance of the Server
  static Server * m_instance;

  ///The ID given to the last Administrator
  int m_last_id_administrator;

  ///The LogFile
  boost::scoped_ptr<LogFile> m_log;

  ///A mutex
  mutable boost::mutex m_mutex;

  ///The parameters when not/yes loaded
  boost::shared_ptr<Parameters> m_parameters;

  //States
  ///The current state
  ServerState * m_state;
  ///The state in the IPGG in which the payoffs for the participants is calculated
  boost::scoped_ptr<ServerStateAssignPayoff> m_state_assign_payoff;
  ///The chatting state
  boost::scoped_ptr<ServerStateChat> m_state_chat;
  ///The state in which the Participant needs to choose an action
  boost::scoped_ptr<ServerStateChooseAction> m_state_choose_action;
  ///The state when the experiment is finished
  boost::scoped_ptr<ServerStateFinished> m_state_finished;
  ///The state when the server is assigning participants to groups
  boost::scoped_ptr<ServerStateGroupDynamics> m_state_group_dynamics;
  ///The state when each Participant can view the results of all the groups
  boost::scoped_ptr<ServerStateViewResultsAll> m_state_view_results_all;
  ///The state when each Participant can view the results within his/her group
  boost::scoped_ptr<ServerStateViewResultsGroup> m_state_view_results_group;
  ///The state when each Participant can view the results of the (majority) voting state
  boost::scoped_ptr<ServerStateViewResultsVoting> m_state_view_results_voting;
  ///The state when each Participant can (majority) vote
  boost::scoped_ptr<ServerStateVoting> m_state_voting;
  ///The state when waiting? What for?
  boost::scoped_ptr<ServerStateWaiting> m_state_waiting;



  ///GetNewGroupNumber returns a number that is not an existing group number
  //int GetNewGroupNumber() const;

  ///The slot for the Observers to listen to
  void OnTimer();

  ///Checks if a file exists
  static bool FileExists(const std::string& filename);

  ///Converts a file to a std::vector
  static const std::vector<std::string> FileToVector(const std::string& filename);

  ///Give friendship to ServerState for the FindParticipant method
  friend class ServerState;
};
//---------------------------------------------------------------------------
#endif // SERVER_H
