//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#ifndef SERVERSTATE_H
#define SERVERSTATE_H
//---------------------------------------------------------------------------
#include <string>
#include <vector>
//---------------------------------------------------------------------------
#include <boost/checked_delete.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/noncopyable.hpp>
//---------------------------------------------------------------------------
#include "forward_declarations.h"
#include "state.h"
//---------------------------------------------------------------------------
///\brief
///ServerState embodies the state of the server.
///
///ServerState is a State.
struct ServerState : public boost::noncopyable, State
{
  ///Create a ServerState with a pointer to its Server
  ServerState(Server * const server);

  ///Check if this state can go to the next state.
  ///This method is implemented like this for possible future
  ///more complex state transitions
  virtual bool CanGoToNextState() const = 0;

  ///Find a read/write Group from a read-only Group
  ///
  ///\note
  ///This is the only method that const_casts a Group
  Group * FindGroup(const Group * const group) const;


  ///Find a read/write Participant from a read-only Participant
  ///
  ///\note
  ///This is the only method that const_casts a Participant
  Participant * FindParticipant(const Participant * const participant) const;

  ///Go to the next state. Each state knows its next state.
  ///This method is implemented like this for possible future
  ///more complex state transitions
  virtual void GoToNextState() const = 0;

  ///Obtain all Participant's their states relevant for the current ServerState
  const std::vector<std::string> GetParticipantStates() const;

  ///Obtain the time left (in seconds) in this State
  int GetTimeLeft() const;

  ///Obtain the States parent
  Server * GetServer() const { return m_server; }

  ///Obtain the duration of the state in seconds
  virtual int GetStateDuration() const = 0;

  ///ServerState dependent response to a timer
  virtual void OnTimer() = 0;

  ///Reset the timer, when the State is started
  void ResetTimeLeft();

  ///Start or restart the state, for example by resetting the timer
  virtual void Start() = 0;


  ///Convert the State to a std::string
  ///virtual const std::string ToStr() const = 0;

  protected:
  ///Only let smart pointers delete this ServerState
  virtual ~ServerState() {}
  ///\brief
  ///Only let smart pointers delete this ServerState
  ///
  ///Do not forget the template brackets, as stated in
  ///Herb Sutter. Exceptional C++ style. 2005. ISBN: 0-201-76042-8. Item 8: 'Befriending templates'.
  friend void boost::checked_delete<>(ServerState*);

  ///(re)set the time left in this state
  void SetTimeLeft(const int time);

  private:
  ///The server this state works on
  Server * const m_server;

  ///A Stopwatch used in the timed States
  const boost::scoped_ptr<Stopwatch> m_stopwatch;

  ///The time that must yet be waited before the state is ended
  int m_wait_time;

  public:
  //Helper functions
  ///Get the heighest integer possible
  static int GetMaxInt();
};
//---------------------------------------------------------------------------
#endif // ServerSTATE_H
