//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <cassert>
#include <iostream>
//---------------------------------------------------------------------------
#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <boost/lambda/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/lexical_cast.hpp>
//---------------------------------------------------------------------------
#include "group.h"
#include "groupfinished.h"
#include "grouploggedin.h"
#include "groupnotloggedin.h"
#include "groupparticipating.h"
#include "groupassigner.h"
#include "groups.h"
#include "state.h"
#include "logfile.h"
#include "parameters.h"
#include "parametersassignpayoff.h"
#include "parameterschat.h"
#include "parameterschooseaction.h"
#include "parametersfinished.h"
#include "parametersgroupdynamics.h"
#include "parametersviewresultsall.h"
#include "parametersviewresultsgroup.h"
#include "parametersviewresultsvoting.h"
#include "parametersvoting.h"
#include "participant.h"
#include "stopwatch.h"
#include "server.h"
#include "serverstateassignpayoff.h"
#include "serverstatechat.h"
#include "serverstatechooseaction.h"
#include "serverstatefinished.h"
#include "serverstategroupdynamics.h"
#include "serverstateviewresultsall.h"
#include "serverstateviewresultsgroup.h"
#include "serverstateviewresultsvoting.h"
#include "serverstatevoting.h"
#include "serverstatewaiting.h"
//---------------------------------------------------------------------------
ServerStateGroupDynamics::ServerStateGroupDynamics(
  Server * const server)
  : ServerState(server)
{

}
//---------------------------------------------------------------------------
///Check if this state can go to the next state.
void ServerStateGroupDynamics::AssignGroup(const Participant * const participant) const
{
  const GroupAssigner * const assigner = participant->GetGroupAssigner();
  assert(assigner);
  if (const GroupAssignerPredetermined * const p
    = dynamic_cast<const GroupAssignerPredetermined*>(assigner))
  {
    const Group * const group_from
      = this->GetServer()->GetGroups()->FindMyGroup(participant);

    const std::vector<const Group*> groups
      = this->GetServer()->GetGroups()->CollectGroups();

    const int group_to_index = p->GetGroup();

    const Group * const group_to
      = *std::find_if(groups.begin(),groups.end(),
        boost::bind(&Group::GetId,boost::lambda::_1) == group_to_index);

    //Transfer the Particippayoffsant
    FindGroup(group_to)->AddParticipant(
      FindGroup(group_from)->RemoveParticipant(participant));



    //Log group assignment
    Server::Get()->GetLog()->LogAssignGroup(
      group_to,
      participant);
  }
  else
  {
    assert(!"?Should not get here?");
  }

  //A new participant should have his/her group assigned, but can also be
  //unassigned from/to a group
  //assert(participant->CanGetGroup() &&
  //  "A new participant should have his/her group assigned");

}
//---------------------------------------------------------------------------
///Check if the Participant can be assigned to a group.
bool ServerStateGroupDynamics::CanAssignGroup(
  const Participant * const  participant) const
{
  //Sure, any logged-in Participant can be assigned to a group
  return GetServer()->GetGroups()->GetGroupLoggedIn()->IsMember(participant);
}
//---------------------------------------------------------------------------
///Check if this state can go to the next state.
///ServerStateGroupDynamics can only go to the next state if
///- all Participants have had their IP addresses assigned
bool ServerStateGroupDynamics::CanGoToNextState() const
{
  assert(GetServer());
  assert(GetServer()->GetParameters());

  return
    //Nobody must be not logged in
    GetServer()->GetGroups()->CollectParticipants(true,false,false,false).empty()
    //Somebody must be logged in or participating
    && !GetServer()->GetGroups()->CollectParticipants(false,true,true,false).empty();
}
//---------------------------------------------------------------------------
///HasFreePlaces is called by GroupAssignerByServer
/*
bool ServerStateGroupDynamics::HasFreePlaces() const
{
  assert(!GetServer()->GetParameters()->GetGroupDynamics()->GetType()
    == ParametersGroupDynamics::static_dynamics
    && "GroupAssignerByServer calls this method, should be static_dynamics");

  return !m_free_group_places.empty();
}
*/
//---------------------------------------------------------------------------
///OccupyFreePlace is called by GroupAssignerByServer
/*
int ServerStateGroupDynamics::OccupyFreePlace()
{
  assert(HasFreePlaces());
  const int group_index = m_free_group_places.back();
  m_free_group_places.pop_back();
  return group_index;
}
*/
//---------------------------------------------------------------------------
///Checks if there is are unassigned participants
//bool ServerStateGroupDynamics::HasUnassignedParticipants() const
//{
//}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
///Obtain the duration of the state in seconds
int ServerStateGroupDynamics::GetStateDuration() const
{
  return this->GetMaxInt();
  //assert(m_parameters);
  //return m_parameters->GetDuration();
}
//---------------------------------------------------------------------------
const std::pair<const Group *,const Group *>
  ServerStateGroupDynamics::GetWorstAndBestGroup() const
{

  const std::map<const Group *,double>& group_payoffs
    = GetServer()->GetStateViewResultsAll()->GetGroupPayoffs();
  assert(group_payoffs.size() >= 2);
  //Quit trying to be smart
  const std::map<const Group *,double>::const_iterator end = group_payoffs.end();
  std::map<const Group *,double>::const_iterator min = group_payoffs.begin();
  std::map<const Group *,double>::const_iterator max = group_payoffs.begin();
  for (std::map<const Group *,double>::const_iterator i = group_payoffs.begin(); i!=end; ++i)
  {
    if (i->second < min->second) min = i;
    if (i->second >= max->second) max = i;
  }
  const std::pair<const Group *,const Group *> p
    = std::make_pair(min->first,max->first);


  /*
  //Check if all values are equal
  {
    const double x = group_payoffs.begin()->second;
    if (std::find_if(
      group_payoffs.begin(),
      group_payoffs.end(),
      boost::lambda::bind(&std::pair<const Group *,double>::second, boost::lambda::_1)
       != x)
       == group_payoffs.end())
     {
       //All values are equal
       const std::pair<const Group *,const Group *>
         p = std::make_pair(
           group_payoffs.begin()->first,
           (--group_payoffs.end())->first);
       assert(p.first != p.second);
       return p;
     }
  }
  //Not all values are equal

  const std::pair<const Group *,const Group *>
    p = std::make_pair(
          std::min_element(
            group_payoffs.begin(),group_payoffs.end(),
            boost::lambda::bind(&std::pair<const Group *,int>::second, boost::lambda::_2)
             > boost::lambda::bind(&std::pair<const Group *,int>::second, boost::lambda::_1)
        )->first,
          std::max_element(
            group_payoffs.rbegin(),group_payoffs.rend(),
            boost::lambda::bind(&std::pair<const Group *,int>::second, boost::lambda::_2)
             > boost::lambda::bind(&std::pair<const Group *,int>::second, boost::lambda::_1)
        )->first
     );
   */
   assert(p.first != p.second);
   return p;
}
//---------------------------------------------------------------------------
///Go to the next state. Each state knows its next state.
void ServerStateGroupDynamics::GoToNextState() const
{
  GetServer()->GetGroups()->MoveLoggedInToParticipating();

  GetServer()->SetState(GetServer()->GetStateChat());
}
//---------------------------------------------------------------------------
///Let this one Group split from 3 Participants
///to 5 Participants
void ServerStateGroupDynamics::NotifyGroupGrowth(const Group * const group)
{
  this->GetServer()->GetGroups()->GrowGroup(group);
  /*
  //yes, unassigned will be changed
  Group * const unassigned
    = GetServer()->FindGroup(
      GetServer()->CollectPassiveGroup());
  //yes, group will be changed
  Group * const group = GetServer()->FindGroup(group);

  assert(unassigned);
  assert(group);

  if (!unassigned->CollectParticipants().empty())
  {
    group->AddParticipant(
      unassigned->RemoveParticipant())
  }
  else
  {
    std::clog << "Warning: no unassigned Participants left to assign to group\n";
  }
  if (!unassigned->CollectParticipants().empty())
  {
    group->AddParticipant(
      unassigned->RemoveParticipant())
  }
  else
  {
    std::clog << "Warning: no unassigned Participants left to assign to group\n";
  }
  */
}
//---------------------------------------------------------------------------
///Let this one Group split from 5 Participants
///to two Groups of 3 Participants
void ServerStateGroupDynamics::NotifyGroupSplit(const Group * const group)
{
  this->GetServer()->GetGroups()->SplitGroup(group);
}
//---------------------------------------------------------------------------
///Kill the Group
void ServerStateGroupDynamics::NotifyKillGroup(const Group * const group)
{
  this->GetServer()->GetGroups()->KillGroup(group);
}
//---------------------------------------------------------------------------
void ServerStateGroupDynamics::OnTimer()
{
  //std::clog << "ServerStateGroupDynamics::OnTimer()\n";
  if (CanGoToNextState()) GoToNextState();
}
//---------------------------------------------------------------------------
void ServerStateGroupDynamics::SetParameters(const ParametersGroupDynamics* const parameters)
{
  assert(parameters);

  //No parameters yet...

}
//---------------------------------------------------------------------------
///Start or restart the state, calculate the joiners
void ServerStateGroupDynamics::Start()
{
  //Set the state of the participants
  BOOST_FOREACH(const Participant * const p,
    GetServer()->GetGroups()->CollectParticipants(false,false,true,false))
  {
    StateGroupDynamics * const state
      = dynamic_cast<const StateGroupDynamics*>(this);
    assert(state);
    FindParticipant(p)->SetState(state);
  }

  if (this->GetServer()->GetParameters()->GetGroupDynamics()->GetType()
    != ParametersGroupDynamics::dynamic)
  {
    return;
  }
  assert(this->GetServer()->GetParameters()->GetGroupDynamics()->GetType()
    == ParametersGroupDynamics::dynamic);

  //Calculate joiners
  const std::map<const Group *,double>& group_payoffs
    = GetServer()->GetStateViewResultsAll()->GetGroupPayoffs();

  if (group_payoffs.empty() || group_payoffs.size() == 1)
  {
    return;
  }


  /*
  const Group * const most_successful_group
    = std::max_element(
      group_payoffs.begin(),
      group_payoffs.end(),
      boost::lambda::bind(&std::pair<const Group *,double>::second, boost::lambda::_2)
       > boost::lambda::bind(&std::pair<const Group *,double>::second, boost::lambda::_1)
     )->first;

  assert(most_successful_group);
  */
  const std::pair<const Group *,const Group *>
    p = GetWorstAndBestGroup();

  /*
  const std::pair<const Group *,const Group *>
    p = std::make_pair(
          std::min_element(
            group_payoffs.begin(),group_payoffs.end(),
            boost::lambda::bind(&std::pair<const Group *,int>::second, boost::lambda::_2)
             > boost::lambda::bind(&std::pair<const Group *,int>::second, boost::lambda::_1)
        )->first,
          std::max_element(
            group_payoffs.rbegin(),group_payoffs.rend(),
            boost::lambda::bind(&std::pair<const Group *,int>::second, boost::lambda::_2)
             > boost::lambda::bind(&std::pair<const Group *,int>::second, boost::lambda::_1)
        )->first
     );
  */
  assert(p.first != p.second);
  const Group * const most_successful_group = p.second;
  //Get the group size of the group with the highest payoffs
  const int group_size = most_successful_group->GetSize();

  if (group_size == 3)
  {
    std::clog << "A group of size 3 grows to 5!\n";
    NotifyGroupGrowth(most_successful_group);
  }
  else if (group_size == 5)
  {
    std::clog << "A group of size 5 splits!\n";
    //Find the group with the highest payoff
    ///\note: this must be done before group split-up,
    ///because otherwise the fresh Group with a payoff of zero
    ///will be deleted immediatly
    //const Group * const least_successful_group
    //  = std::min_element(
    //    group_payoffs.rbegin(),
    //    group_payoffs.rend(),
    //    boost::lambda::bind(&std::pair<const Group *,double>::second, boost::lambda::_2)
    //     > boost::lambda::bind(&std::pair<const Group *,double>::second, boost::lambda::_1)
    //   )->first;
    const Group * const least_successful_group
      = p.first;
    assert(least_successful_group);
    assert(least_successful_group != most_successful_group);
    ///\note
    ///The operation below invalidates const Group * const most_successful_group
    NotifyKillGroup(least_successful_group);
    //Again find the most successful group and let it split
    //Find the group with the highest payoff
    assert(most_successful_group->GetSize() == 5);

    NotifyGroupSplit(most_successful_group);

  }
}
//---------------------------------------------------------------------------
