//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#ifndef SERVERSTATEGROUPDYNAMICS_H
#define SERVERSTATEGROUPDYNAMICS_H
//---------------------------------------------------------------------------
#include <map>
//---------------------------------------------------------------------------
#include <boost/shared_ptr.hpp>
//---------------------------------------------------------------------------
#include "serverstate.h"
//---------------------------------------------------------------------------
///\brief
///The Server in the state when each Participant is assigned his/her group
struct ServerStateGroupDynamics : public ServerState, StateGroupDynamics
{
  ///Create this ServerState by passing the Server it is a State of
  ServerStateGroupDynamics(Server * const server);

  ///Request the server to assign the Participant to a Group
  void AssignGroup(const Participant * const participant) const;

  ///Ask the server if the Participant can be asssigned to a Group
  bool CanAssignGroup(const Participant* const participant) const;

  ///HasFreePlaces is called by GroupAssignerByServer
  //bool HasFreePlaces() const;

  ///OccupyFreePlace is called by GroupAssignerByServer
  //int OccupyFreePlace();

  ///Check if this state can go to the next state.
  ///This method is implemented like this for possible future
  ///more complex state transitions
  bool CanGoToNextState() const;

  ///Obtain all Participant's their states relevant for the current ServerState
  //const std::vector<std::string> GetParticipantStates() const;

  ///Checks if there is at least one Participant that is unassigned
  //bool HasUnassignedParticipants() const;

  ///Go to the next state. Each state knows its next state.
  ///This method is implemented like this for possible future
  ///more complex state transitions
  void GoToNextState() const;

  ///Let this one Group split from 3 Participants
  ///to 5 Participants
  void NotifyGroupGrowth(const Group * const group);

  ///Let this one Group split from 5 Participants
  ///to two Groups of 3 Participants
  void NotifyGroupSplit(const Group * const group);

  ///Kill the Group
  void NotifyKillGroup(const Group * const group);

  ///ServerState dependent response to a timer
  void OnTimer();

  ///Obtain the duration of the state in seconds
  int GetStateDuration() const;

  ///Create this ServerState its specific Parameters
  void SetParameters(const ParametersGroupDynamics* const);

  ///Set the ID's (=key) of the Participants that are
  ///allowed to join a group (=value)
  //void SetJoiners(const std::map<int,int>& joiners);

  ///Start or restart the state, calculate the joiners
  void Start();

  ///Represent this ServerState as a std::string
  const std::string ToStr() const { return this->StateGroupDynamics::ToStr(); }

  private:
  ///Only allow a Boost smart pointer to delete ServerStateGroupDynamics
  //to prevent the following trouble,
  //cited from http://www.boost.org/libs/utility/checked_delete.html:
  //The C++ Standard allows, in 5.3.5/5, pointers to incomplete
  //class types to be deleted with a delete-expression.
  //When the class has a non-trivial destructor, or a class-specific operator
  //delete, the behavior is undefined. Some compilers issue a warning when an
  //incomplete type is deleted, but unfortunately, not all do, and programmers
  //sometimes ignore or disable warnings.
  ~ServerStateGroupDynamics() {}
  ///Only allow a Boost smart pointer to delete ServerStateGroupDynamics
  //Template syntax from Herb Sutter. Exceptional C++ style. 2005. ISBN: 0-201-76042-8. Item 8: 'Befriending templates'.
  friend void boost::checked_delete<>(ServerStateGroupDynamics*);

  const std::pair<const Group *,const Group *> GetWorstAndBestGroup() const;

  ///m_free_group_places are the groups that can be joined
  //std::vector<int> m_free_group_places;

  ///Get the number of Participants in a group
  //int GetGroupSize(const int group_number) const;

  ///Get the number of the least succesful group
  //int GetNumberLeastSuccessfulGroup() const;

  ///Get the number of the most succesful group
  //int GetNumberMostSuccessfulGroup() const;
};
//---------------------------------------------------------------------------
#endif // SERVERSTATEGROUPDYNAMICS_H
