//---------------------------------------------------------------------------
/*
GTST, Game Theory Server
Copyright (C) 2011 Richel Bilderbeek

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
//---------------------------------------------------------------------------
//From http://www.richelbilderbeek.nl/ProjectGtst.htm
//---------------------------------------------------------------------------
#include <cassert>
#include <iostream>
#include <numeric>
//---------------------------------------------------------------------------
#include <boost/foreach.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/numeric/conversion/cast.hpp>
//---------------------------------------------------------------------------
#include "group.h"
#include "groupfinished.h"
#include "grouploggedin.h"
#include "groupnotloggedin.h"
#include "groupparticipating.h"
#include "groups.h"
#include "state.h"
#include "logfile.h"
#include "parameters.h"
#include "parametersassignpayoff.h"
#include "parameterschat.h"
#include "parameterschooseaction.h"
#include "parametersfinished.h"
#include "parametersgroupdynamics.h"
#include "parametersviewresultsall.h"
#include "parametersviewresultsgroup.h"
#include "parametersviewresultsvoting.h"
#include "parametersvoting.h"
#include "participant.h"
#include "repeatassigner.h"
#include "stopwatch.h"
#include "server.h"
#include "serverstateassignpayoff.h"
#include "serverstatechat.h"
#include "serverstatechooseaction.h"
#include "serverstatefinished.h"
#include "serverstategroupdynamics.h"
#include "serverstateviewresultsall.h"
#include "serverstateviewresultsgroup.h"
#include "serverstateviewresultsvoting.h"
#include "serverstatevoting.h"
#include "serverstatewaiting.h"
//---------------------------------------------------------------------------
ServerStateViewResultsAll::ServerStateViewResultsAll(
  Server * const server)
  : ServerState(server),
    m_parameters(0),
    m_repeats_left(-1)
{

}
//---------------------------------------------------------------------------
///Check if this state can go to the next state.
bool ServerStateViewResultsAll::CanGoToNextState() const
{
  return GetTimeLeft() < 0;
}
//---------------------------------------------------------------------------
///Go to the next state. Each state knows its next state.
void ServerStateViewResultsAll::GoToNextState() const
{
  assert(m_repeats_left >= 0);
  if (CanGoToNextState())
  {
    if (m_repeats_left == 0)
    {
      GetServer()->SetState(GetServer()->GetStateFinished());
    }
    else
    {
      GetServer()->SetState(GetServer()->GetStateGroupDynamics());
    }
  }
}
//---------------------------------------------------------------------------
const std::vector<std::string> ServerStateViewResultsAll::GetResults(
  const Participant * const participant) const
{
  assert(GetServer());
  assert(GetServer()->GetParameters());
  assert(GetServer()->GetParameters()->GetChooseAction());
  assert(participant);
  const std::vector<double>& payoffs = participant->GetPayoffs();
  assert(!payoffs.empty());

  const Group * const group = GetServer()->GetGroups()->FindMyGroup(participant);

  std::vector<std::string> v;
  v.push_back("All results");
  v.push_back(" ");
  v.push_back(std::string("You are in group #")
    + boost::lexical_cast<std::string>(group->GetId()));
  v.push_back(std::string("Your last payoff is ")
    + boost::lexical_cast<std::string>(payoffs.back()));
  v.push_back(std::string("Your total payoff is ")
    + boost::lexical_cast<std::string>(
    std::accumulate(payoffs.begin(),payoffs.end(),0.0)));
  v.push_back(" ");
  v.push_back(" ");
  v.push_back("Group total collected payoff:");


  BOOST_FOREACH(const Group* const any_group,
    GetServer()->GetGroups()->CollectGroups(false,false,true,false))
  {
    assert(!any_group->GetAveragePayoffs().empty());

    std::string s
      = std::string("Group #")
      + boost::lexical_cast<std::string>(any_group->GetId())
      + std::string(": ")
      + boost::lexical_cast<std::string>(any_group->GetAveragePayoffs().back())
      + (any_group == group
        ? std::string(" (YOUR GROUP)")
        : std::string(""));
    v.push_back(s);
  }
  return v;
}
//---------------------------------------------------------------------------
///Obtain the duration of the state in seconds
int ServerStateViewResultsAll::GetStateDuration() const
{
  assert(m_parameters);
  return m_parameters->GetDuration();
}
//---------------------------------------------------------------------------
void ServerStateViewResultsAll::OnTimer()
{
  if (CanGoToNextState()) GoToNextState();
}
//---------------------------------------------------------------------------
void ServerStateViewResultsAll::SetParameters(const ParametersViewResultsAll* const parameters)
{
  assert(parameters);
  m_parameters = parameters;
  m_repeats_left = m_parameters->GetNumberOfPeriods();
  assert(m_repeats_left >= 0
    && "m_repeats must be zero or positive");

  ResetTimeLeft();
  assert(this->GetTimeLeft() >= 0);
}
//---------------------------------------------------------------------------
///\brief
///Start or restart the state
///
///For ServerStateViewResultsAll the following is done:
///- Calculate each group its payoff from the last IPGG
void ServerStateViewResultsAll::Start()
{
  //Update m_group_payoffs
  const int n_ipgg_cycles = GetServer()->GetParameters()->GetViewResultsGroup()->GetRepeatAssigner()->GetLastAssigned();

  {
    m_group_payoffs.clear();
    assert(m_group_payoffs.empty());
    const std::vector<const Group *> v
      = GetServer()->GetGroups()->CollectGroups(false,false,true,false);
    BOOST_FOREACH(const Group * const g,v)
    {
      double payoff = 0.0;
      //Sum the payoff
      const std::vector<const Participant *> w = g->CollectParticipants();
      BOOST_FOREACH(const Participant * p,w)
      {
        payoff
          += std::accumulate(
            p->GetPayoffs().end()-n_ipgg_cycles,
            p->GetPayoffs().end(),
            0.0);
      }
      //Average the payoff
      payoff /= boost::numeric_cast<double>(w.size());
      m_group_payoffs.insert(std::make_pair(g,payoff));
      FindGroup(g)->AppendAveragePayoff(payoff);
    }
  }
  assert(m_group_payoffs.size() == GetServer()->GetGroups()->CollectGroups(false,false,true,false).size());
  /*
  //std::map<int,int> group_sizes;

  //Calculate the sum group benefits from the last IPGG cycle

  BOOST_FOREACH(const Participant * const participant,
    GetServer()->CollectActiveParticipants())
  {
    if (participant->CanGetGroup())
    {
      const int group = participant->GetGroup();

      const std::vector<double>& payoffs = participant->GetPayoffs();

      #ifdef DEBUG_2784283456783483
      if (!(n_last_payoffs < boost::numeric_cast<int>(payoffs.size())))
      {
        std::clog
          << "n_last_payoffs " << n_last_payoffs << '\n'
          << "payoffs.size " << payoffs.size() << '\n';
      }
      #endif

      assert(n_last_payoffs <= boost::numeric_cast<int>(payoffs.size()));

      const double payoff = std::accumulate(payoffs.end()-n_last_payoffs,payoffs.end(),0.0);

      //Count the group member
      if (group_sizes.find(group) == group_sizes.end())
      {
        group_sizes[group] = 0;
      }
      ++group_sizes[group];

      //Sum the payoff
      if (m_group_payoffs.find(group) == m_group_payoffs.end())
      {
        m_group_payoffs[group] = 0.0;
      }
      m_group_payoffs[group] += payoff;
    }
  }
  //Calculate the average group payoffs
  {
    typedef std::pair<const int,double> Pair;
    BOOST_FOREACH(Pair & p,m_group_payoffs)
    {
      const int group = p.first;
      assert(group_sizes.find(group) != group_sizes.end());
      const int size = group_sizes.find(group)->first;
      #define DEBUG_08723872763542763587257654765
      #ifdef  DEBUG_08723872763542763587257654765
      std::clog << "size: " << size << '\n';
      #endif
      assert(size != 0);
      p.second/=boost::numeric_cast<double>(size);
    }
  }
  */

  //Keep track of the program flow
  assert(m_repeats_left >= 0
    && "m_repeats must be zero or positive");
  if (m_repeats_left == 0)
  {
    assert(m_parameters);
    m_repeats_left = m_parameters->GetNumberOfPeriods();
  }
  --m_repeats_left;
  assert(m_repeats_left >= 0
    && "m_repeats must be zero or positive");

  BOOST_FOREACH(const Participant * const p,
    GetServer()->GetGroups()->CollectParticipants(false,false,true,false))
  {
    StateViewResultsAll * const state
      = dynamic_cast<const StateViewResultsAll*>(this);
    assert(state);
    FindParticipant(p)->SetState(state);
  }
}
//---------------------------------------------------------------------------
