//
// connection.hpp
// ~~~~~~~~~~~~~~
//
// Copyright (c) 2003-2012 Christopher M. Kohlhoff (chris at kohlhoff dot com)
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//

#ifndef SERIALIZATION_CONNECTION_HPP
#define SERIALIZATION_CONNECTION_HPP

#include <boost/asio.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/archive/text_oarchive.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/tuple/tuple.hpp>
#include <boost/signals2.hpp>
#include <iomanip>
#include <string>
#include <sstream>
#include <vector>

class connection {
private:
	template <typename T>
	struct value{
		T _t;
		void operator()(const T& t, const boost::system::error_code& e){
			_t = t;
		}
	};

public:
	connection(boost::asio::io_service& io_service);

	~connection();

	boost::asio::ip::tcp::socket& socket();

	template <typename T, typename Handler>
	void async_write(const T& t, Handler handler) {
		std::vector<boost::asio::const_buffer> buffers = build_buffers(t);
		boost::asio::async_write(_socket, buffers, handler);
	}

	template <typename T>
	void sync_write(const T& t) {
		std::vector<boost::asio::const_buffer> buffers = build_buffers(t);
		boost::asio::write(_socket, buffers);
	}

	template <typename T>
	std::vector<boost::asio::const_buffer> build_buffers(const T& t){
		std::ostringstream archive_stream;
		boost::archive::text_oarchive archive(archive_stream);
		archive << t;
		_outbound_data = archive_stream.str();

		std::ostringstream header_stream;
		header_stream << std::setw(header_length) << std::hex << _outbound_data.size();
		std::vector<boost::asio::const_buffer> buffers;
		if (!header_stream || header_stream.str().size() != header_length)
		{
			std::cerr << "Invalid argument, header not proper " << header_stream.str().size() << ", " << header_length << std::endl;
			return buffers;
		}
		_outbound_header = header_stream.str();

		buffers.push_back(boost::asio::buffer(_outbound_header));
		buffers.push_back(boost::asio::buffer(_outbound_data));

		return buffers;
	}

	template <typename T, typename Handler>
	void async_read(Handler handler) {
		void (connection::*f)( const boost::system::error_code&, boost::tuple<Handler>) = &connection::handle_read_header<T, Handler>;
		boost::asio::async_read(_socket, boost::asio::buffer(_inbound_header),
				boost::bind(f, this, boost::asio::placeholders::error, boost::make_tuple(handler)));
	}

	template <typename T>
	T sync_read(){
		boost::system::error_code e;
		boost::asio::read(_socket, boost::asio::buffer(_inbound_header), e);
		value<T> v;
		if (interpret_header(e)){
			boost::asio::read(_socket, boost::asio::buffer(_inbound_data));
			handle_read_data<T>(e, boost::make_tuple(boost::ref(v)));
		}

		return v._t;
	}

	boost::signals2::signal<void()>& destroyed();
	boost::signals2::signal<void()>& closed();

	void close();
//	void shutdown(boost::asio::ip::tcp::socket::shutdown_type shutdown_type);


private:
	template <typename T, typename Handler>
	void handle_read_header(const boost::system::error_code& e, boost::tuple<Handler> handler) {
		if (interpret_header(e)){
			void (connection::*f)( const boost::system::error_code&, boost::tuple<Handler>) = &connection::handle_read_data<T, Handler>;
			boost::asio::async_read(_socket, boost::asio::buffer(_inbound_data), boost::bind(f, this, boost::asio::placeholders::error, handler));
		}
	}

	bool interpret_header(const boost::system::error_code& e);

	template <typename T, typename Handler>
	void handle_read_data(const boost::system::error_code& e, boost::tuple<Handler> handler) {
		if (e) {
			std::cerr << e.message() << std::endl;
		}
		else {
			try {
				T t;
				std::string archive_data(&_inbound_data[0], _inbound_data.size());
				std::istringstream archive_stream(archive_data);
				boost::archive::text_iarchive archive(archive_stream);
				archive >> t;
				boost::get<0>(handler)(t, e);
			}
			catch (std::exception& e) {
				std::cerr << "Invalid argument" << std::endl;
				std::cerr << e.what() << std::endl;
				return;
			}

		}
	}

private:
	boost::asio::ip::tcp::socket _socket;

	enum { header_length = 8 };

	std::string _outbound_header;

	std::string _outbound_data;

	char _inbound_header[header_length];

	std::vector<char> _inbound_data;

	boost::signals2::signal<void()> _destroyed;

	boost::signals2::signal<void()> _closed;
};

typedef boost::shared_ptr<connection> connection_ptr;


#endif // SERIALIZATION_CONNECTION_HPP
