#pragma once

#include <Wt/WCompositeWidget>
#include <Wt/WSignal>

namespace Wt {

class WText;
class WLineEdit;
class WPushButton;

/*! \class WInPlaceEdit Wt/WInPlaceEdit Wt/WInPlaceEdit
 *  \brief A widget that provides in-place-editable text.
 *
 * The %WInPlaceEdit provides a text that may be edited in place by
 * the user by clicking on it. When clicked, the text turns into a
 * line edit, with optionally a save and cancel button (see
 * setButtonsEnabled()).
 *
 * When the user saves the edit, the valueChanged() signal is emitted.
 *
 * Usage example:
 * \if cpp
 * \code
 * Wt::WContainerWidget *w = new Wt::WContainerWidget();
 * new Wt::WText("Name: ", w);
 * Wt::WInPlaceEdit *edit = new Wt::WInPlaceEdit("Bob Smith", w);
 * edit->setStyleClass("inplace");
 * \endcode
 * \elseif java
 * \code 
 * WContainerWidget w = new WContainerWidget();
 * new WText("Name: ", w);
 * WInPlaceEdit edit = new WInPlaceEdit("Bob Smith", w);
 * edit.setStyleClass("inplace");
 * \endcode
 * \endif
 *
 * This code will produce an edit that looks like:
 * \image html WInPlaceEdit-1.png "WInPlaceEdit text mode"
 * When the text is clicked, the edit will expand to become:
 * \image html WInPlaceEdit-2.png "WInPlaceEdit edit mode"
 *
 * <h3>CSS</h3>
 *
 * A WInPlaceEdit widget renders as a <tt>&lt;span&gt;</tt> containing
 * a WText, a WLineEdit and optional buttons (WPushButton). All these
 * widgets may be styled as such. It does not provide style information.
 *
 * In particular, you may want to provide a visual indication that the text
 * is editable e.g. using a hover effect:
 *
 * CSS stylesheet:
 * \code
 * .inplace span:hover {
 *    background-color: gray;
 * }
 * \endcode
 */
class WInPlaceAreaEdit : public WCompositeWidget
{
public:
  /*! \brief Creates an in-place edit with the given text.
   */
  WInPlaceAreaEdit(const WString& text, WContainerWidget *parent = 0);

  /*! \brief Returns the current value.
   *
   * \sa setText()
   */
  const WString& text() const;

  /*! \brief Sets the current value.
   *
   * \sa text()
   */
  void setText(const WString& text);

  /*! \brief Sets the empty text to be shown when the field is empty 
   *         and not being edited.
   *
   * \sa emptyText()
   */
  void setEmptyText(const WString& emptyText);

  /*! \brief Returns the empty text to be shown when the field is empty 
   *         and not being edited.
   *
   * \sa setEmptyText()
   */
  const WString& emptyText();

  /*! \brief Returns the line edit.
   *
   * You may use this for example to set a validator on the line edit.
   */
  WTextArea *textArea() const { return edit_; }

  /*! \brief Returns the WText widget that renders the current string.
   *
   * You may use this for example to set the text format of the displayed
   * string.
   */
  WText *textWidget() const { return text_; }

  /*! \brief Returns the save button.
   *
   * This method returns \c 0 if the buttons were disabled.
   *
   * \sa cancelButton(), setButtonsEnabled()
   */
  WPushButton *saveButton() const { return save_; }

  /*! \brief Returns the cancel button.
   *
   * This method returns \c 0 if the buttons were disabled.
   *
   * \sa saveButton(), setButtonsEnabled()
   */
  WPushButton *cancelButton() const { return cancel_; }

  /*! \brief %Signal emitted when the value has been changed.
   *
   * The signal argument provides the new value.
   */
  Signal<WString>& valueChanged() { return valueChanged_; }

  /*! \brief Displays the Save and 'Cancel' button during editing
   *
   * By default, the Save and Cancel buttons are shown. Call this
   * function with \p enabled = \c false to only show a line edit.
   *
   * In this mode, the enter key or any event that causes focus to be
   * lost saves the value while the escape key cancels the editing.
   */
  void setButtonsEnabled(bool enabled = true);

private:
  void save();
  void cancel();

private:
  Signal<WString> valueChanged_;
  WContainerWidget *impl_, *editing_;
  WText            *text_;
  WString          emptyText_;
  WTextArea        *edit_;
  WPushButton      *save_;
  WPushButton      *cancel_;
  Wt::Signals::connection c1_;
  Wt::Signals::connection c2_;
  bool              empty_;
};

}
